import pytest

from idbroker.utils import (
    get_configured_schools,
    group_dn_has_configured_school,
    group_event_on_configured_schools,
    role_has_configured_school,
    school_in_configured_schools,
    user_event_on_configured_schools,
)
from ucsschool_id_connector.models import ListenerUserOldDataEntry
from ucsschool_id_connector.utils import school_class_dn_regex, workgroup_dn_regex


@pytest.mark.parametrize(
    "group_dn,configured_schools,expected",
    [
        (
            "cn=fOObar-1a,cn=klassen,cn=schueler,cn=groups,ou=fOObar,dc=traeger2,dc=test",
            {"foobar", "world"},
            True,
        ),
        ("cn=fOObar-wg1,cn=schueler,cn=groups,ou=fOObar,dc=traeger2,dc=test", {"foobar", "world"}, True),
        ("cn=ocean-1a,cn=klassen,cn=groups,ou=ocean,dc=traeger2,dc=test", {"foobar", "world"}, False),
        (
            "cn=ocean-wg1,cn=schueler,cn=schueler,cn=groups,ou=ocean,dc=traeger2,dc=test",
            {"foobar", "world"},
            False,
        ),
        ("cn=fOObar-wg1,cn=schueler,cn=groups,ou=fOObar,dc=traeger2,dc=test", {"*", "world"}, True),
        ("cn=fOObar-wg1,cn=schueler,cn=groups,ou=fOObar,dc=traeger2,dc=test", {"f*b*r", "world"}, True),
        ("cn=fOObar-wg1,cn=schueler,cn=groups,ou=fOObar,dc=traeger2,dc=test", {"z*r", "world"}, False),
    ],
)
def test_group_dn_has_configured_school(group_dn, configured_schools, expected, monkeypatch):
    monkeypatch.setenv("ldap_base", "dc=traeger2,dc=test")
    workgroup_dn_regex.cache_clear()
    school_class_dn_regex.cache_clear()
    assert group_dn_has_configured_school(group_dn, configured_schools) == expected


@pytest.mark.parametrize(
    "group_dn,configured_schools,expected",
    [
        (
            "cn=fOObar-1a,cn=klassen,cn=schueler,cn=groups,ou=fOObar,dc=traeger2,dc=test",
            {"foobar", "world"},
            True,
        ),
        ("cn=fOObar-wg1,cn=schueler,cn=groups,ou=fOObar,dc=traeger2,dc=test", {"foobar", "world"}, True),
        ("cn=ocean-1a,cn=klassen,cn=groups,ou=ocean,dc=traeger2,dc=test", {"foobar", "world"}, False),
        (
            "cn=ocean-wg1,cn=schueler,cn=schueler,cn=groups,ou=ocean,dc=traeger2,dc=test",
            {"foobar", "world"},
            False,
        ),
        ("cn=fOObar-wg1,cn=schueler,cn=groups,ou=fOObar,dc=traeger2,dc=test", {"*", "world"}, True),
        ("cn=fOObar-wg1,cn=schueler,cn=groups,ou=fOObar,dc=traeger2,dc=test", {"f*b*r", "world"}, True),
        ("cn=fOObar-wg1,cn=schueler,cn=groups,ou=fOObar,dc=traeger2,dc=test", {"z*r", "world"}, False),
    ],
)
def test_group_event_on_configured_schools(
    group_dn,
    configured_schools,
    expected,
    monkeypatch,
    get_listener_group_add_modify_object,
    get_listener_group_remove_object,
):
    monkeypatch.setenv("ldap_base", "dc=traeger2,dc=test")
    workgroup_dn_regex.cache_clear()
    school_class_dn_regex.cache_clear()
    listener_obj = get_listener_group_add_modify_object(dn=group_dn)
    assert group_event_on_configured_schools(listener_obj, configured_schools) == expected
    listener_obj = get_listener_group_remove_object(dn=group_dn)
    assert group_event_on_configured_schools(listener_obj, configured_schools) == expected


@pytest.mark.parametrize(
    "config_schools,expected",
    [
        (["foobar", "testi"], {"foobar", "testi"}),
        (["foobar", "testi", "testi"], {"foobar", "testi"}),
        (["foobar", "testi", "tEsTi"], {"foobar", "testi"}),
        (["foobar", "t*T*i", "tEsTi"], {"foobar", "t*t*i", "testi"}),
        ([], set()),
    ],
)
def test_get_configured_schools(config_schools, expected, school_authority_conf):
    school_authority_conf.plugin_configs["id_broker"]["schools"] = config_schools
    schools = get_configured_schools(school_authority_conf)
    assert schools == expected


@pytest.mark.parametrize(
    "school,configured_schools,expected",
    [
        ("foobar", {"foobar", "testi"}, True),
        ("foobar", {"bar", "testi"}, False),
        ("foobar", {"*", "testi"}, True),
        ("foobar", {"f*b*r", "testi"}, True),
        ("bar", {"f*b*r", "testi"}, False),
        ("foobar", {"foobar*", "testi"}, True),
    ],
)
def test_school_in_configured_schools(school, configured_schools, expected):
    assert school_in_configured_schools(school, configured_schools) is expected


@pytest.mark.parametrize(
    "role_string,configured_schools,expected",
    [
        ("student:school:fOObar", {"foobar", "world"}, True),
        ("student:school:fOObar", {"*", "world"}, True),
        ("student:school:fOObar", {"f*b*r", "world"}, True),
        ("student:school:fOObar", {"z*r", "world"}, False),
        ("student:school:ocean", {"foobar", "world"}, False),
        ("student:soccer_team:foobar", {"foobar", "world"}, False),
    ],
)
def test_role_has_configured_school(role_string, configured_schools, expected):
    assert role_has_configured_school(role_string, configured_schools) == expected


@pytest.mark.parametrize(
    "config_schools,new_schools,old_schools,expected",
    [
        (
            {"foobar", "testi"},
            ["foobar", "testi", "fake"],
            [],
            True,
        ),
        ({"foobar"}, ["FoOBaR"], [], True),
        ({"*"}, ["FoOBaR"], [], True),
        ({"*"}, [], ["FoOBaR"], True),
        ({"f*b*r"}, ["FoOBaR"], [], True),
        ({"z*r"}, ["FoOBaR"], [], False),
        ({"foobar"}, ["FoOBaR"], None, True),
        ({"dogs"}, ["cats"], ["dogs"], True),
        (set(), ["cats"], ["dogs"], False),
        (set(), ["foobar"], [], False),
        (set(), [], ["foobar"], False),
        ({"bar"}, ["foobar", "testi"], [], False),
        ({"bar"}, [], [], False),
        ({"bar", "testi"}, ["foobar", "testi"], [], True),
    ],
)
def test_user_event_on_configured_schools(
    config_schools,
    new_schools,
    old_schools,
    expected,
    get_listener_user_add_modify_object,
    get_listener_user_remove_object,
    user_dn,
):
    if new_schools:
        listener_obj = get_listener_user_add_modify_object(
            dn=user_dn,
            object={
                "school": new_schools,
            },
        )
    else:
        listener_obj = get_listener_user_remove_object(dn=user_dn)
    if old_schools is not None:
        listener_obj.old_data = ListenerUserOldDataEntry(schools=old_schools)
    school_in_config = user_event_on_configured_schools(listener_obj, config_schools)
    assert school_in_config is expected
