#!/usr/bin/env python3
#
# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

import asyncio
import datetime
import json
import logging
import os
import sys
import time
from contextlib import asynccontextmanager
from typing import Any, Dict, List

import click
from fastapi import HTTPException

try:
    from idbroker.config_utils import CONFIG_PATH, NoConfigFoundError, read_id_broker_config
    from idbroker.id_broker_client import IDBrokerSchool
    from idbroker.initial_sync import get_ous, setup, sync_complete
    from idbroker.utils import get_configured_schools, school_in_configured_schools
except ImportError:  # pragma: no cover
    sys.path.append(
        "/var/lib/univention-appcenter/apps/ucsschool-id-connector/conf/plugins/packages"
    )  # pragma: no cover
    from idbroker.initial_sync import (
        setup,
        sync_complete,
        get_ous,
    )  # pragma: no cover
    from idbroker.config_utils import (
        read_id_broker_config,
        CONFIG_PATH,
        NoConfigFoundError,
    )  # pragma: no cover
    from idbroker.id_broker_client import IDBrokerSchool  # pragma: no cover
    from idbroker.utils import get_configured_schools, school_in_configured_schools

from ucsschool_id_connector.http_api import query_service
from ucsschool_id_connector.models import (
    SchoolAuthorityConfiguration,
    SchoolAuthorityConfigurationPatchDocument,
)
from ucsschool_id_connector.school_scheduler import SchoolScheduler
from ucsschool_id_connector.utils import ConsoleAndFileLogging


class ManageSchoolsException(Exception):
    ...


class ManageSchoolsOnIDBroker:
    wait_school_deletion_timeout = 500
    wait_school_deletion_interval = 3
    queue_check_timeout = 500
    queue_check_interval = 3

    def __init__(
        self,
        school_authority: str,
        logger: logging.Logger,
        force: bool,
        num_tasks_for_initial_sync: int = 0,
    ):
        self.num_tasks = num_tasks_for_initial_sync
        self.school_authority = school_authority
        self.logger = logger
        self.school_scheduler = SchoolScheduler()
        self.force = force
        setup_logging(
            loggers=[
                self.logger,
                self.school_scheduler.logger,
                self.school_scheduler.user_scheduler.logger,
                self.school_scheduler.group_scheduler.logger,
            ]
        )

    def get_config(self) -> SchoolAuthorityConfiguration:
        return read_id_broker_config(
            config_path=CONFIG_PATH, school_authority_name=self.school_authority
        )

    def get_schools_from_config(self):
        config = self.get_config().dict()
        return config["plugin_configs"]["id_broker"].get("schools", [])

    @asynccontextmanager
    async def config_transaction(self):
        old_school_authority_config = self.get_config()
        try:
            yield
        except BaseException:
            self.logger.warning("Rollback last config change.")
            await self.restore_old_config(old_school_authority_config)
            raise

    async def restore_old_config(self, old_config: SchoolAuthorityConfiguration):
        config = old_config.dict_secrets_as_str()
        try:
            await self.patch_school_authority_config(data={"plugin_configs": config["plugin_configs"]})
        except ManageSchoolsException:
            self.logger.error(
                "Failed to restore settings via HTTP API. Restoring manually."
                " Restart the UCS@school ID Connector."
            )
            school_authority_config_path = os.path.join(
                CONFIG_PATH, os.path.basename(f"{self.school_authority}.json")
            )
            with open(school_authority_config_path, "w") as fout:
                json.dump(fp=fout, obj=config)

    async def add_school_to_config(self, school: str):
        schools = self.get_schools_from_config()
        _lower_schools = [school.lower() for school in schools]
        if school.lower() in _lower_schools:
            if self.force is False:
                raise ManageSchoolsException(
                    f"School {school} was already added (use --force if you want to resync the school)"
                )
            # save with same case as provided in cli argument
            schools = [s for s in schools if s.lower() != school.lower()]
        schools.append(school)
        patch_data = {"plugin_configs": {"id_broker": {"schools": schools}}}
        await self.patch_school_authority_config(
            data=patch_data,
        )

    async def patch_school_authority_config(self, data: Dict[str, Any]):
        patch_data = SchoolAuthorityConfigurationPatchDocument(**data)
        try:
            res = await query_service(
                cmd="patch_school_authority", name=self.school_authority, school_authority=patch_data
            )
            if res.get("errors"):
                raise ManageSchoolsException(f"Error while trying to patch the school authority: {res}")
        except HTTPException as e:
            raise ManageSchoolsException(f"Error while trying to patch the school authority: {e!r}")

    async def set_initial_sync_mode(self, value: bool):
        patch_data = {"plugin_configs": {"id_broker": {"initial_import_mode": value}}}
        await self.patch_school_authority_config(
            data=patch_data,
        )

    async def remove_school_from_config(self, school: str):
        schools = self.get_schools_from_config()
        _lower_schools = [s.lower() for s in schools]
        if school.lower() not in _lower_schools:
            if not self.force:
                raise ManageSchoolsException(
                    f"School {school} was already removed from configuration. "
                    "Use --force if you want to retry deleting the school from the ID Broker."
                )
            elif self.force and "*" in school:
                # We only support wildcards in the ID Broker plugin, not for removing configuration keys
                raise ManageSchoolsException(
                    "Wildcards are not supported in combination with --force. "
                    "Use --all_schools if you want to delete all schools. "
                    "Or temporarily add the school to ensure you keep a valid configuration."
                )
            else:
                self.logger.info(f"School {school} was already removed from configuration.")
                return
        schools = [s for s in schools if s.lower() != school.lower()]
        patch_data = {"plugin_configs": {"id_broker": {"schools": schools}}}
        await self.patch_school_authority_config(data=patch_data)

    async def wait_for_empty_queue(self):
        start = time.time()
        while True:
            res = await query_service(cmd="get_queue", name=self.school_authority)
            if res.get("errors"):
                raise ManageSchoolsException(
                    f"Error while trying to get the length of queue for school authority: {self.school_authority}"
                )
            queue_length = res["result"]["length"]
            if queue_length > 0:
                self.logger.info(f"{queue_length} Remaining objects.")
                if time.time() > start + self.queue_check_timeout:
                    self.logger.warning(
                        f"Not all elements of the queue were processed after {self.queue_check_timeout}s. "
                        "Please check the ID Connector is configured correctly."
                        " Otherwise continue waiting."
                    )
                await asyncio.sleep(self.queue_check_interval)
            else:
                return

    def check_config_is_active_for_initial_sync(self):
        if not self.get_config().active:
            raise ManageSchoolsException("Config needs to be active for initial sync")

    async def add_schools(
        self,
        schools: List[str],
    ):
        """Add the school to the config and schedule all existing users & groups.
        If the school is not existing, the second step won't do anything."""

        synced_schools = set()

        for school in schools:
            async with self.config_transaction():
                ldap_schools = await get_ous(ou=school)
                if not ldap_schools:
                    raise ManageSchoolsException(
                        f"School {school} does not exist on this system. "
                        "Use '--initial_sync False' if you want to add the school anyway."
                    )

                self.logger.info(f"Connect school {school} to the ID Broker.")
                self.logger.info(f"Adding {school} to the schools which are synced to the ID Broker.")
                self.check_config_is_active_for_initial_sync()
                await self.add_school_to_config(school=school)
                await self.set_initial_sync_mode(value=True)
                setup(config_path=CONFIG_PATH)
                for ldap_school in ldap_schools:
                    if ldap_school.name in synced_schools:
                        self.logger.info(f"School {ldap_school.name} has already been synced this run.")
                        continue
                    self.logger.info(f"Sync groups of school {ldap_school.name} (without members).")
                    await sync_complete(
                        school=ldap_school.name,
                        dry_run=False,
                        max_workers=self.num_tasks,
                        sync_members=False,
                    )
                    self.logger.info(
                        f"Sync users of school {ldap_school.name} without groups using the ID Connector. "
                        "This might take a while."
                    )
                    await self.school_scheduler.queue_school(
                        school=ldap_school.name, num_tasks=self.num_tasks
                    )
                    self.logger.info(
                        f"Waiting until all users of {ldap_school.name} are synced. This will take a while."
                    )
                    await self.wait_for_empty_queue()
                    self.logger.info(f"Update group memberships of school {ldap_school.name}.")
                    await sync_complete(
                        school=ldap_school.name,
                        dry_run=False,
                        max_workers=self.num_tasks,
                        sync_members=True,
                    )
                    synced_schools.add(ldap_school.name)
                await self.set_initial_sync_mode(value=False)
        self.logger.info("Done adding schools.")

    async def remove_schools(self, schools: List[str], delete_schools: bool):
        for school in schools:
            self.logger.info(f"Removing school {school}.")
            async with self.config_transaction():
                await self.remove_school_from_config(school=school)
                if delete_schools:
                    await self.remove_schools_on_id_broker(school_filter=school)
        self.logger.info("Done removing schools.")

    async def configure_schools(self, schools: List[str] = None):
        self.logger.info(
            f"Adding {len(schools)} schools to the schools which are synced to the ID Broker."
        )
        self.logger.info(f"{[school for school in schools]}")
        for school in schools:
            async with self.config_transaction():
                await self.add_school_to_config(school=school)

    async def remove_schools_on_id_broker(self, school_filter: str):
        school_client = IDBrokerSchool(
            school_authority=self.get_config(),
            plugin_name="id_broker",
        )
        self.logger.info(
            f"Deleting schools matching {school_filter} "
            f"for school authority {self.school_authority} on the ID Broker."
        )
        ldap_schools = await get_ous(school_filter)
        if not ldap_schools:
            # We need the school id to be able to delete the school.
            self.logger.error(f"School {school_filter} does not exist on the ID Broker.")
            sys.exit(1)

        configured_schools = get_configured_schools(self.get_config())
        for ldap_school in ldap_schools:
            if school_in_configured_schools(ldap_school.name.lower(), configured_schools):
                # This can happend in in combination with wildcards
                self.logger.warning(
                    f"School {ldap_school.name} is still configured and will not be deleted."
                )
                continue
            await school_client.delete(ldap_school.id)
            end = datetime.datetime.now() + datetime.timedelta(seconds=self.wait_school_deletion_timeout)
            while True:
                school_exists = await school_client.exists(ldap_school.id)
                if school_exists is False:
                    break
                else:
                    if datetime.datetime.now() > end:
                        self.logger.warning(
                            f"The school with name {ldap_school.name} might be still existing after"
                            f" {self.wait_school_deletion_timeout}s on the ID Broker."
                            "Removing large schools can take a long time."
                        )
                self.logger.info(
                    f"Waiting for the school {ldap_school.name} to be deleted on the ID Broker."
                )
                await asyncio.sleep(self.wait_school_deletion_interval)
        self.logger.info(f"Successfully deleted school {school_filter}.")


@click.group()
def manage_schools_cli():
    pass  # pragma: no cover


@manage_schools_cli.command(
    name="remove_schools",
    help="Remove schools from the set of schools which are synchronized to the ID Broker "
    "and remove them from the ID Broker system."
    "The SCHOOLS argument must match the configured schools value in the school authority configuration.",
)
@click.option("-a", "--school_authority", required=True, type=str)
@click.option(
    "-f",
    "--force",
    is_flag=True,
    default=False,
    help="Add this option to also handle already removed schools.",
)
@click.option(
    "--all_schools",
    is_flag=True,
    default=False,
    help="Add this option to remove all schools.",
)
@click.option(
    "--delete_schools",
    default=True,
    help="Setting this to false only removes the school from the configuration "
    "without removing it from the ID Broker.",
)
@click.argument("schools", required=False, nargs=-1)
def remove_schools(
    schools: List[str], school_authority: str, force: bool, all_schools: bool, delete_schools: bool
):
    logger = ConsoleAndFileLogging.get_logger(__name__)
    manager = ManageSchoolsOnIDBroker(school_authority=school_authority, logger=logger, force=force)
    if not all_schools and not schools:
        logger.error("Missing positional argument schools or --all_schools option.")
        sys.exit(1)
    if all_schools and schools:
        logger.error("Passing schools with --all_schools option is not permitted.")
        sys.exit(1)
    try:
        if all_schools:
            schools = manager.get_schools_from_config()
            if len(schools) == 0:
                logger.error("No schools are configured.")
                sys.exit(1)
        asyncio.run(
            manager.remove_schools(
                schools=schools,
                delete_schools=delete_schools,
            )
        )
    except (ManageSchoolsException, NoConfigFoundError, KeyboardInterrupt) as exc:
        if isinstance(exc, KeyboardInterrupt):
            error_msg = "Aborted"
        else:
            error_msg = str(exc)
        logger.error(f"An error was raised while trying to remove the schools: {error_msg}")
        sys.exit(1)


def setup_logging(loggers: List[logging.Logger]):
    for logger in loggers:
        ConsoleAndFileLogging.add_console_handler(logger)


@manage_schools_cli.command(
    name="add_schools",
    help="Configure SCHOOLS to the ID Broker and synchronize the respective school objects."
    '"*" Can be used as a wildcard character. '
    "E.g. if you want to sync all your current and future schools.",
)
@click.option(
    "-t",
    "--num_tasks",
    type=click.IntRange(1, 32),
    default=1,
    help="The number of parallel tasks used for running the initial sync.",
)
@click.option("-a", "--school_authority", required=True, type=str)
@click.option(
    "-i",
    "--initial_sync",
    type=bool,
    default=True,
    help="Synchronize objects of the given schools actively to the ID Broker "
    "and configure them to the ID Broker."
    "If set to false, existing schools are configured to the ID Broker,"
    " but the school objects are not actively synchronized. "
    "To apply this for all schools, omid the schools argument."
    "If the schools do not yet exist on the ID Broker system, "
    "they need to be triggered manually by using the schedule_school script.",
)
@click.option(
    "-f",
    "--force",
    is_flag=True,
    default=False,
    help="Add this option to also handle already synchronized schools.",
)
@click.option(
    "--all_schools",
    is_flag=True,
    default=False,
    help="Add this option to configure all current schools.",
)
@click.argument("schools", nargs=-1)
def add_schools(
    num_tasks: int,
    schools: List[str],
    school_authority: str,
    initial_sync: bool,
    force: bool,
    all_schools: bool,
):
    logger = ConsoleAndFileLogging.get_logger(__name__)
    manager = ManageSchoolsOnIDBroker(
        school_authority=school_authority,
        logger=logger,
        num_tasks_for_initial_sync=num_tasks,
        force=force,
    )
    if not all_schools and not schools:
        logger.error("Missing positional argument schools or --all_schools option.")
        sys.exit(1)
    if all_schools:
        if schools:
            logger.error("Passing schools with --all_schools option is not permitted.")
            sys.exit(1)
        schools = [school.name for school in asyncio.run(get_ous("*"))]
    try:
        if initial_sync:
            asyncio.run(manager.add_schools(schools=schools))
        else:
            asyncio.run(manager.configure_schools(schools=schools))
    except (ManageSchoolsException, NoConfigFoundError, KeyboardInterrupt) as exc:
        if isinstance(exc, KeyboardInterrupt):
            error_msg = "Aborted"
        else:
            error_msg = str(exc)
        logger.error(f"An error was raised while trying to add the schools: {error_msg}")
        sys.exit(1)


if __name__ == "__main__":
    manage_schools_cli()  # pragma: no cover
