#!/usr/bin/env python3
#
# -*- coding: utf-8 -*-
#
# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

import asyncio
import logging
import os
import sys
import time
from argparse import ArgumentParser, ArgumentTypeError, RawDescriptionHelpFormatter
from datetime import datetime
from typing import Callable, List, Optional

import lazy_object_proxy
from ldap3 import Entry
from tenacity import Retrying, retry_if_not_exception_type, stop_after_attempt, wait_fixed

from ucsschool_id_connector.ldap_access import LDAPAccess
from ucsschool_id_connector.models import SchoolAuthorityConfiguration
from ucsschool_id_connector.utils import ConsoleAndFileLogging

try:
    from config_utils import CONFIG_PATH, read_id_broker_config
    from verify import (
        dn_to_id,
        ldap_object_to_school,
        ldap_object_to_school_class,
        ldap_object_to_workgroup,
    )

    from .provisioning_api.exceptions import ApiException
except ImportError:
    sys.path.append("/var/lib/univention-appcenter/apps/ucsschool-id-connector/conf/plugins/packages")
    from idbroker.config_utils import CONFIG_PATH, read_id_broker_config
    from idbroker.provisioning_api.exceptions import ApiException
    from idbroker.verify import (
        dn_to_id,
        ldap_object_to_school,
        ldap_object_to_school_class,
        ldap_object_to_workgroup,
    )

try:
    # works in IDE, but not on the cli
    from .id_broker_client import (
        IDBrokerError,
        IDBrokerObject,
        IDBrokerSchool,
        IDBrokerSchoolClass,
        IDBrokerUser,
        IDBrokerWorkGroup,
        School,
        SchoolClass,
        WorkGroup,
    )
except ImportError:
    # works on the cli in ID Connector Docker container
    # make `idbroker` package importable
    import sys

    sys.path.append("/var/lib/univention-appcenter/apps/ucsschool-id-connector/conf/plugins/packages")

    from idbroker.id_broker_client import (
        IDBrokerError,
        IDBrokerObject,
        IDBrokerSchool,
        IDBrokerSchoolClass,
        IDBrokerUser,
        IDBrokerWorkGroup,
        School,
        SchoolClass,
        WorkGroup,
    )

SCHOOL_ATTRS = ["entryUUID", "ou", "displayName"]
GROUP_ATTRS = ["entryUUID", "cn", "description", "uniqueMember"]

MAX_THREADS = 32


logger: logging.Logger = lazy_object_proxy.Proxy(
    lambda: ConsoleAndFileLogging.get_logger(os.path.basename(__file__))
)
ldap = lazy_object_proxy.Proxy(lambda: LDAPAccess())

DEFAULT_POOL_SIZE: int = 4
auth_config: Optional[SchoolAuthorityConfiguration] = None
USER_CLIENT: Optional[IDBrokerUser] = None
SCHOOL_CLIENT: Optional[IDBrokerSchool] = None
SCHOOL_CLASS_CLIENT: Optional[IDBrokerSchoolClass] = None
WORKGROUP_CLIENT: Optional[IDBrokerWorkGroup] = None
DEFAULT_RETRY_SLEEP_TIME: int = 10
TASK_SEM: Optional[asyncio.Semaphore] = None


async def limited_func(
    sem: asyncio.Semaphore, func: Callable, sync_members: bool, dry_run: bool, obj: IDBrokerObject
):
    async with sem:
        return await func(obj=obj, sync_members=sync_members, dry_run=dry_run)


async def parallelize(
    funcs, objs, sync_members: bool, dry_run: bool, max_workers: int = DEFAULT_POOL_SIZE
):
    task_limiter = asyncio.Semaphore(max_workers)
    tasks = [
        limited_func(sync_members=sync_members, dry_run=dry_run, sem=task_limiter, func=func, obj=obj)
        for func in funcs
        for obj in objs
    ]
    return await asyncio.gather(*tasks)


async def get_workgroups(
    ou: str,
) -> List[WorkGroup]:
    ldap_filter = f"(&(ucsschoolRole=workgroup:school:{ou})(memberUid=*))"
    logger.info("Retrieving workgroups with filter %r...", ldap_filter)
    objects: List[Entry] = await ldap.search(filter_s=ldap_filter, attributes=GROUP_ATTRS)
    return [ldap_object_to_workgroup(o) for o in objects]


async def get_classes(
    ou: str,
) -> List[SchoolClass]:
    ldap_filter = f"(&(ucsschoolRole=school_class:school:{ou})(memberUid=*))"
    logger.info("Retrieving classes with filter %r...", ldap_filter)
    objects: List[Entry] = await ldap.search(filter_s=ldap_filter, attributes=GROUP_ATTRS)
    return [ldap_object_to_school_class(o) for o in objects]


async def get_ous(
    ou: str,
) -> List[School]:
    ldap_filter = f"(&(objectClass=ucsschoolOrganizationalUnit)(ou={ou}))"
    logger.info("Retrieving OUs with filter %r...", ldap_filter)
    objects: List[Entry] = await ldap.search(
        base=os.environ.get("ldap_base"), filter_s=ldap_filter, attributes=SCHOOL_ATTRS
    )
    return [ldap_object_to_school(o) for o in objects]


async def sync_complete(school: str, dry_run: bool, sync_members: bool, max_workers: int) -> None:
    ous = await get_ous(school)
    logger.info("Found %d schools.", len(ous))
    school_classes = await get_classes(school)
    logger.info("Found %d classes.", len(school_classes))
    workgroups = await get_workgroups(school)
    logger.info("Found %d workgroups.", len(workgroups))
    t0 = time.time()
    await parallelize(
        funcs=[sync_object], objs=ous, max_workers=1, sync_members=sync_members, dry_run=dry_run
    )
    if not dry_run:
        logger.info("Synchronized %d schools in %0.2f minutes.", len(ous), (time.time() - t0) / 60.0)
    t0 = time.time()
    await parallelize(
        funcs=[sync_object],
        objs=school_classes,
        max_workers=max_workers,
        sync_members=sync_members,
        dry_run=dry_run,
    )
    if not dry_run:
        logger.info(
            "Synchronized %d classes in %0.2f minutes.", len(school_classes), (time.time() - t0) / 60.0
        )
    t0 = time.time()
    await parallelize(
        funcs=[sync_object],
        objs=workgroups,
        max_workers=max_workers,
        sync_members=sync_members,
        dry_run=dry_run,
    )
    if not dry_run:
        logger.info(
            "Synchronized %d workgroups in %0.2f minutes.", len(workgroups), (time.time() - t0) / 60.0
        )


async def handle_group_members(members_dns, sync_members: bool):
    if sync_members and members_dns:
        _member_ids = [await dn_to_id(member) for member in members_dns]
        member_ids = []
        # Check if members are existing on target system.
        for member_id in _member_ids:
            if await USER_CLIENT.exists(member_id):
                member_ids.append(member_id)
            else:
                logger.warning(f"User with id={member_id} does not exist on ID Broker. Skip it.")
        return member_ids
    else:
        return []


async def update_group_members(client, obj: IDBrokerObject, name: str, dry_run: bool):
    if not dry_run:
        logger.info("Updating %s %r...", type(obj), name)
        for attempt in Retrying(
            stop=stop_after_attempt(3),
            wait=wait_fixed(10),
            retry=retry_if_not_exception_type(IDBrokerError),
        ):
            with attempt:
                await client.update(obj)
    else:
        logger.info("[dry-run] Updating %s %r...", type(obj), name)


async def sync_object(obj: IDBrokerObject, sync_members: bool, dry_run: bool) -> None:
    obj_type = type(obj)
    if obj_type in (School, SchoolClass, WorkGroup):
        obj_type = type(obj)
    else:
        raise ValueError(f"Could not identify type of given object {obj.name}")
    client = {School: SCHOOL_CLIENT, SchoolClass: SCHOOL_CLASS_CLIENT, WorkGroup: WORKGROUP_CLIENT}[
        obj_type
    ]
    if obj_type in [SchoolClass, WorkGroup]:
        name = f"{obj.school}-{obj.name}"
        obj.members = await handle_group_members(members_dns=obj.members, sync_members=sync_members)
    else:
        name = obj.name

    logger.info("Checking %s %r...", obj_type.__name__, name)
    for attempt in Retrying(
        stop=stop_after_attempt(3), wait=wait_fixed(10), retry=retry_if_not_exception_type(IDBrokerError)
    ):
        with attempt:
            exists = await client.exists(obj.id)
    try:
        if exists and sync_members and obj_type in (SchoolClass, WorkGroup):
            await update_group_members(client=client, obj=obj, name=name, dry_run=dry_run)
        else:
            if not dry_run and not exists:
                logger.info("Creating %s %r...", obj_type.__name__, name)
                for attempt in Retrying(
                    stop=stop_after_attempt(3),
                    wait=wait_fixed(10),
                    retry=retry_if_not_exception_type(IDBrokerError),
                ):
                    with attempt:
                        await client.create(obj)
            elif dry_run and not exists:
                logger.info("[dry-run] Creating %s %r...", obj_type.__name__, name)
    except (asyncio.exceptions.TimeoutError, ApiException, IDBrokerError) as exc:
        logger.info("ERROR handling %s %r: %s.", obj_type.__name__, name, str(exc))


def setup(config_path: str) -> None:
    global SCHOOL_CLIENT, SCHOOL_CLASS_CLIENT, USER_CLIENT, WORKGROUP_CLIENT, auth_config
    auth_config = read_id_broker_config(config_path=config_path)
    USER_CLIENT = IDBrokerUser(school_authority=auth_config, plugin_name="id_broker")
    SCHOOL_CLIENT = IDBrokerSchool(school_authority=auth_config, plugin_name="id_broker")
    SCHOOL_CLASS_CLIENT = IDBrokerSchoolClass(school_authority=auth_config, plugin_name="id_broker")
    WORKGROUP_CLIENT = IDBrokerWorkGroup(school_authority=auth_config, plugin_name="id_broker")


def num_task_range():
    def checker(arg):
        if arg.isdigit():
            tasks = int(arg)
        else:
            raise ArgumentTypeError(
                "Invalid argument type! The entered value is not type int or negative."
            )
        if tasks < 1:
            raise ArgumentTypeError("At least one task is needed to run.")
        if tasks > MAX_THREADS:
            raise ArgumentTypeError(
                f"The maximum allowed value for concurrent tasks is {MAX_THREADS} (Entered: {arg})"
            )
        return tasks

    return checker


if __name__ == "__main__":
    parser = ArgumentParser(description=__doc__, formatter_class=RawDescriptionHelpFormatter)
    parser.add_argument(
        "-d",
        "--dry-mode",
        help="fakes complete sync with no commitment to the IDBroker and displays which objects would be created",
        action="store_true",
    )
    parser.add_argument(
        "-m",
        "--members",
        help="enables the sync of users",
        action="store_true",
    )
    parser.add_argument(
        "-s",
        "--school",
        help="only sync a single school with its groups",
        type=str,
    )
    parser.add_argument(
        "-p",
        "--parallelism",
        type=num_task_range(),
        default=DEFAULT_POOL_SIZE,
        help=f"defines the number of parallel tasks that are used for the specified task (min=1, "
        f"max={MAX_THREADS})",
    )
    parser.add_argument(
        "-c",
        "--config_path",
        default=CONFIG_PATH,
        help="defines the path where the configuration for school_authorities can be found",
    )

    parser.epilog = """Examples:
    $ initial_sync.py
        syncs all ous and groups without members (add -m to include members)

    $ initial_sync.py -s OU:str
        syncs all groups of the ou specified by OU without members (add -m to include members)

    $ initial_sync.py -d
        fake sync as preview for all ous and groups without members (add -m to include members)

    $ initial_sync.py -d -s OU:str
        fake sync of singular ou specified by OU without members (add -m to include members)

    """

    opt = parser.parse_args()
    handler = logging.StreamHandler(stream=sys.stdout)
    handler.setFormatter(ConsoleAndFileLogging.get_formatter())
    logger.addHandler(handler)
    logger.info("STARTING improved first time sync with arguments %s...", opt)
    logger.info("Logging to '/var/log/univention/ucsschool-id-connector/queues.log'.")
    start = datetime.now()
    logger.info("Running with %d tasks.", int(opt.parallelism))

    setup(config_path=opt.config_path)
    school = str(opt.school) if opt.school else "*"
    asyncio.run(
        sync_complete(
            school=school,
            dry_run=opt.dry_mode,
            max_workers=int(opt.parallelism),
            sync_members=opt.members,
        )
    )

    end = datetime.now()
    duration = (end - start).total_seconds()
    logger.info("DONE improved first time sync. Processed in: %r seconds", duration)
