# -*- coding: utf-8 -*-

# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

"""
Client for the ID Broker Provisioning API.
"""
import abc
import datetime
import json
import logging
import os
import re
import uuid
from typing import Dict, List, Match, Optional, Type, Union, cast

import jwt
import lazy_object_proxy
from async_property import async_property
from pydantic import BaseModel, validator

from ucsschool_id_connector.models import SchoolAuthorityConfiguration
from ucsschool_id_connector.utils import ConsoleAndFileLogging

from .provisioning_api import (
    ApiClient,
    ApiException,
    AuthApi,
    Configuration as GenConfiguration,
    School as GenSchool,
    SchoolClass as GenSchoolClass,
    SchoolClassesApi as GenSchoolClassesApi,
    SchoolContext as GenSchoolContext,
    SchoolsApi as GenSchoolsApi,
    Token as GenToken,
    User as GenUser,
    UsersApi as GenUsersApi,
    WorkGroup as GenWorkGroup,
    WorkgroupsApi as GenWorkGroupsApi,
)

logger: logging.Logger = lazy_object_proxy.Proxy(lambda: ConsoleAndFileLogging.get_logger(__name__))
_shared_token: Dict[str, "Token"] = {}

GenApiObject = Union[GenSchool, GenSchoolClass, GenSchoolContext, GenUser, GenWorkGroup]
GenApiHandler = Union[GenSchoolsApi, GenSchoolClassesApi, GenUsersApi, GenWorkGroupsApi]
IDBrokerObject = Union["School", "SchoolClass", "SchoolContext", "User", "WorkGroup"]
IDBrokerObjectType = Union[
    Type["School"], Type["SchoolClass"], Type["SchoolContext"], Type["User"], Type["WorkGroup"]
]

ID_BROKER_CLIENT_DEFAULT_TIMEOUT = 180.0
CLIENT_TOKEN_LEEWAY = 30


def get_conflict_id_from_exc(exc):
    return json.loads(exc.body)["detail"]["conflict_id"]


def exc_includes_conflict_id(exc):
    try:
        get_conflict_id_from_exc(exc)
    except (json.JSONDecodeError, KeyError, TypeError):
        return False
    return True


def _get_shared_token(key: str) -> Optional["Token"]:
    return _shared_token.get(key, None)


def _set_shared_token(token: "Token", key: str) -> None:
    global _shared_token
    _shared_token[key] = token


class IDBrokerError(Exception):
    def __init__(self, status: int, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.status = status


class IDBrokerNotFoundError(IDBrokerError):
    status = 404


class IDBrokerObjectBase(BaseModel):
    _gen_class: Type[GenApiObject]

    @classmethod
    def from_gen_obj(cls, gen_obj: GenApiObject) -> BaseModel:
        """Convert OpenAPI client object to pydantic object."""
        return cls(**gen_obj.to_dict())

    def to_gen_obj(self) -> GenApiObject:
        """Convert pydantic object to OpenAPI client object."""
        return self._gen_class(**self.dict())

    def __eq__(self, other):
        return self.dict() == other.dict()


class School(IDBrokerObjectBase):
    id: str
    name: str
    display_name: str = ""
    _gen_class = GenSchool

    def __repr__(self):
        return f"{self.__class__.__name__}(name={self.name!r}, id={self.id!r})"

    @validator("display_name", pre=True)
    def none_to_str(cls, value: Optional[str]):
        return "" if value is None else value


class SchoolClass(IDBrokerObjectBase):
    id: str
    name: str
    description: str = ""
    school: str
    members: List[str] = []
    _gen_class = GenSchoolClass

    def __eq__(self, other):
        if not isinstance(other, SchoolClass):
            return False
        return all(
            (
                self.id == other.id,
                self.name == other.name,
                self.description == other.description,
                self.school == other.school,
                set(self.members) == set(other.members),
            )
        )

    def __repr__(self):
        return f"{self.__class__.__name__}(name={self.name!r}, id={self.id!r})"

    @validator("description", pre=True)
    def none_to_str(cls, value: Optional[str]):
        return "" if value is None else value


class WorkGroup(IDBrokerObjectBase):
    id: str
    name: str
    description: str = ""
    school: str
    members: List[str] = []
    _gen_class = GenWorkGroup

    def __eq__(self, other):
        if not isinstance(other, WorkGroup):
            return False
        return all(
            (
                self.id == other.id,
                self.name == other.name,
                self.description == other.description,
                self.school == other.school,
                set(self.members) == set(other.members),
            )
        )

    def __repr__(self):
        return f"{self.__class__.__name__}(name={self.name!r}, id={self.id!r})"

    @validator("description", pre=True)
    def none_to_str(cls, value: Optional[str]):
        return "" if value is None else value


class SchoolContext(IDBrokerObjectBase):
    classes: List[str]
    workgroups: List[str]
    roles: List[str]
    _gen_class = GenSchoolContext

    def __eq__(self, other):
        if not isinstance(other, SchoolContext):
            return False
        return all(
            (
                {c.lower() for c in self.classes} == {c.lower() for c in other.classes},
                {w.lower() for w in self.workgroups} == {w.lower() for w in other.workgroups},
                {r.lower() for r in self.roles} == {r.lower() for r in other.roles},
            )
        )


class User(IDBrokerObjectBase):
    id: str
    first_name: str
    last_name: str
    user_name: str
    legal_guardians: list[str]
    legal_wards: list[str]
    context: Dict[str, SchoolContext]
    _gen_class = GenUser

    def __eq__(self, other):
        if not isinstance(other, User):
            return False
        return all(
            (
                self.id == other.id,
                self.first_name == other.first_name,
                self.last_name == other.last_name,
                self.user_name == other.user_name,
                (not self.legal_wards and not other.legal_wards)
                or (set(self.legal_wards) == set(other.legal_wards)),
                (not self.legal_wards and not other.legal_wards)
                or (set(self.legal_guardians) == set(other.legal_guardians)),
                self.context == other.context,
            )
        )

    def __repr__(self):
        return f"{self.__class__.__name__}(user_name={self.user_name!r}, id={self.id!r})"

    @classmethod
    def from_gen_obj(cls, gen_obj: GenApiObject) -> "User":
        gen_obj = cast(GenUser, gen_obj)
        res = super().from_gen_obj(gen_obj)
        res = cast("User", res)
        res.context = {
            k: SchoolContext(classes=v.classes, roles=v.roles, workgroups=v.workgroups)
            for k, v in gen_obj.context.items()
        }
        return res

    def to_gen_obj(self) -> GenUser:
        res = super().to_gen_obj()
        res = cast(GenUser, res)
        res.context = {
            k: GenSchoolContext(classes=v.classes, roles=v.roles, workgroups=v.workgroups)
            for k, v in self.context.items()
        }
        return res


class Token:
    def __init__(self, api_configuration: GenConfiguration):
        self._configuration = api_configuration
        self._token: Optional[GenToken] = None
        self._token_expiry: Optional[datetime.datetime] = None

    @async_property
    async def access_token(self) -> str:
        """Get the access token, refreshed if required."""
        if not self._token or not self._token_is_valid():
            self._token = await self._fetch_token()
            self._token_expiry = self._token_expiration(self._token.access_token)
            if self._token_expiry < datetime.datetime.utcnow():
                raise ValueError(
                    f"Retrieved expired token. Token expiry is: {self._token_expiry.isoformat()} UTC. "
                    f"Current time is {datetime.datetime.utcnow().isoformat()} UTC."
                )
        return self._token.access_token

    @staticmethod
    def _token_expiration(access_token: str) -> datetime.datetime:
        """Get the time at which `access_token` expires."""
        try:
            payload = jwt.decode(
                access_token,
                algorithms=["HS256"],
                options={"verify_exp": False, "verify_signature": False},
            )
        except jwt.PyJWTError as exc:
            raise ValueError(f"Error decoding token ({access_token!r}): {exc!s}")
        if not isinstance(payload, dict) or "exp" not in payload:
            raise ValueError(f"Payload in token not a dict or missing 'exp' entry ({access_token!r}).")
        try:
            return datetime.datetime.utcfromtimestamp(payload["exp"])
        except ValueError as exc:
            raise ValueError(
                f"Error parsing date {payload['exp']!r} in token ({access_token!r}): {exc!s}"
            )

    def _token_is_valid(self) -> bool:
        if not self._token_expiry or not self._token:
            return False
        if (
            datetime.datetime.utcnow() + datetime.timedelta(seconds=CLIENT_TOKEN_LEEWAY)
            > self._token_expiry
        ):
            return False
        return True

    async def _fetch_token(self) -> GenToken:
        logger.debug("Retrieving token...")
        if not self._configuration.verify_ssl:
            logger.warning("SSL verification is disabled.")
        async with ApiClient(self._configuration) as api_client:
            api_instance = AuthApi(api_client)
            api_response = await api_instance.login_for_access_token_ucsschool_apis_auth_token_post(
                username=self._configuration.username,
                password=self._configuration.password,
                grant_type="password",
            )
            return api_response


class ProvisioningAPIClient(abc.ABC):
    API_METHODS: Dict[str, str]
    PROVISIONING_URL_REGEX = r"^https://(?P<host>.+?)/"
    _object_type: IDBrokerObjectType
    _gen_api_handler: Type[GenApiHandler]
    _share_token = True  # whether all client instances should use the same Token instance

    def __init__(self, school_authority: SchoolAuthorityConfiguration, plugin_name: str):
        self.school_authority = school_authority
        self.plugin_name = plugin_name
        m: Optional[Match] = re.match(self.PROVISIONING_URL_REGEX, school_authority.url)
        if not m:
            raise ValueError(
                f"Bad ID Broker Provisioning URL in school authority configuration {school_authority!r}:"
                f" {school_authority.url!r}. Correct form is: 'https://FQDN/'."
            )
        host = m.groupdict()["host"]
        target_url = f"https://{host}"
        try:
            self.school_authority_name = school_authority.name
            username = school_authority.plugin_configs[plugin_name]["username"]
            password = school_authority.plugin_configs[plugin_name]["password"].get_secret_value()
            version = school_authority.plugin_configs[plugin_name]["version"]
        except KeyError as exc:
            raise ValueError(
                f"Missing {exc!s} in ID Broker Provisioning plugin configuration of school authority: "
                f"{school_authority.dict()!r}."
            )
        if version != 1:
            raise ValueError(f"Unsupported ID Broker Provisioning API version {version!r}.")
        self.configuration = GenConfiguration(host=target_url, username=username, password=password)
        http_proxy = os.environ.get("http_proxy") or os.environ.get("HTTP_PROXY")
        https_proxy = os.environ.get("https_proxy") or os.environ.get("HTTPS_PROXY")
        if https_proxy and not http_proxy:
            logger.warning("Environment variabel HTTPS_PROXY not supported. Use HTTP_PROXY instead.")
        self.configuration.proxy = http_proxy
        self.configuration.verify_ssl = "UNSAFE_SSL" not in os.environ
        if not self.configuration.verify_ssl:
            logger.warning("SSL verification is disabled.")
        shared_token = _get_shared_token(key=f"{username}@{host}")
        if self._share_token and shared_token:
            self.token = shared_token
        else:
            self.token = Token(self.configuration)
            if self._share_token:
                _set_shared_token(token=self.token, key=f"{username}@{host}")

    async def create(self, obj: IDBrokerObject) -> IDBrokerObject:
        raise NotImplementedError

    async def exists(self, obj_id: str) -> bool:
        raise NotImplementedError

    async def get(self, obj_id: str) -> IDBrokerObject:
        raise NotImplementedError

    async def delete(self, obj_id: str) -> None:
        raise NotImplementedError

    async def _create(self, obj_arg_name: str, **kwargs) -> IDBrokerObject:
        """
        Create object.

        `obj_arg_name` is the key to the object to create in kwargs that has to be used in the API call.
        Returned value is the object actually created by the server.
        """
        obj = cast(IDBrokerObject, kwargs.pop(obj_arg_name))
        request_id = uuid.uuid4().hex
        _request_id = request_id[:10]
        logger.debug("[%s] Creating %r...", _request_id, obj)
        gen_obj = obj.to_gen_obj()
        kwargs[obj_arg_name] = gen_obj
        try:
            new_obj = await self._request("post", request_id=request_id, **kwargs)
        except ApiException as exc:
            if exc.status == 409 and exc_includes_conflict_id(exc):
                conflict_id = get_conflict_id_from_exc(exc)
                logger.warn(
                    "[%s] Obj %r has a conflict with %s, trying to delete the conflicting object.",
                    _request_id,
                    obj,
                    conflict_id,
                )
                await self._request(
                    "delete",
                    id=conflict_id,
                    request_id=request_id,
                    school_authority=self.school_authority_name,
                )
                new_obj = await self._request("post", request_id=request_id, **kwargs)
            else:
                exc_cls = IDBrokerNotFoundError if exc.status == 404 else IDBrokerError
                raise exc_cls(
                    exc.status,
                    f"[{_request_id}] Error HTTP {exc.status} ({exc.reason}) creating "
                    f"{self._object_type.__name__} {gen_obj!r}.",
                )
        if not new_obj:
            raise RuntimeError(
                f"[{_request_id}] Empty response creating {self._object_type.__name__} object {gen_obj!r}."
            )
        logger.debug("[%s] Created %r", _request_id, new_obj)
        if obj != new_obj:
            logger.warning(
                "[%s] Requested %s to be created and object returned by server differ. "
                "Requested object:\n%r, returned object:\n%r",
                _request_id,
                self._object_type.__name__,
                obj.dict(),
                new_obj.dict(),
            )
        return new_obj

    async def _delete(self, obj_id: str, **kwargs) -> None:
        """
        Delete object with ID `obj_id`.
        `kwargs` will be passed on to API call.
        """
        request_id = uuid.uuid4().hex
        _request_id = request_id[:10]
        logger.debug("[%s] Deleting %s %r...", _request_id, self._object_type.__name__, obj_id)
        try:
            await self._request("delete", id=obj_id, request_id=request_id, **kwargs)
            logger.debug("[%s] %s %r deleted.", _request_id, self._object_type.__name__, obj_id)
        except ApiException as exc:
            if exc.status != 404:
                raise IDBrokerError(
                    exc.status,
                    f"{_request_id} Error HTTP {exc.status} ({exc.reason}) deleting {self._object_type.__name__} "
                    f"{obj_id!r}.",
                )
            logger.info(
                "[%s] %s %r not deleted, as it did not exist.",
                _request_id,
                self._object_type.__name__,
                obj_id,
            )

    async def _exists(self, obj_id: str, **kwargs) -> bool:
        """
        Check if object with ID `obj_id` exists on the server.
        `kwargs` will be passed on to API call.
        """
        request_id = uuid.uuid4().hex
        _request_id = request_id[:10]
        logger.debug(
            "[%s] Checking existence of %s %r...", _request_id, self._object_type.__name__, obj_id
        )
        try:
            await self._request("head", id=obj_id, request_id=request_id, **kwargs)
            logger.debug("[%s] %s %r exists.", _request_id, self._object_type.__name__, obj_id)
        except ApiException as exc:
            if exc.status != 404:
                raise IDBrokerError(
                    exc.status,
                    f"[{_request_id}] Error HTTP {exc.status} ({exc.reason}) checking existence of "
                    f"{self._object_type.__name__} using {kwargs!r}.",
                )
            logger.debug("[%s] %s %r does not exist.", _request_id, self._object_type.__name__, obj_id)
            return False
        return True

    async def _get(self, obj_id: str, **kwargs) -> IDBrokerObject:
        """
        Retrieve the object with ID `obj_id` from the server.
        `kwargs` will be passed on to API call.
        """
        request_id = uuid.uuid4().hex
        _request_id = request_id[:10]
        logger.debug(
            "[%s] Retrieving %s %r with kwargs %r...",
            _request_id,
            self._object_type.__name__,
            obj_id,
            kwargs,
        )
        try:
            obj = await self._request("get", id=obj_id, request_id=request_id, **kwargs)
        except ApiException as exc:
            exc_cls = IDBrokerNotFoundError if exc.status == 404 else IDBrokerError
            raise exc_cls(
                exc.status,
                f"[{_request_id}] Error HTTP {exc.status} ({exc.reason}) retrieving {self._object_type.__name__} "
                f"using {kwargs!r}.",
            )
        if not obj:
            raise RuntimeError(
                f"[{_request_id}] Empty response retrieving {self._object_type.__name__} object with {kwargs!r}."
            )
        logger.debug("[%s] Retrieved %r.", _request_id, obj)
        return obj

    async def _update(self, obj_arg_name: str, **kwargs) -> IDBrokerObject:
        """
        Modify object on the server

        `obj_arg_name` is the key to the object to modify that will be put into `kwargs`.
        `kwargs` will be passed on to API call.
        Returned value is the object actually created by the server.
        """
        obj = cast(IDBrokerObject, kwargs.pop(obj_arg_name))
        request_id = uuid.uuid4().hex
        _request_id = request_id[:10]
        logger.debug("[%s] Updating %r...", _request_id, obj)
        gen_obj = obj.to_gen_obj()
        kwargs[obj_arg_name] = gen_obj
        try:
            new_obj = await self._request("put", request_id=request_id, **kwargs)
        except ApiException as exc:
            exc_cls = IDBrokerNotFoundError if exc.status == 404 else IDBrokerError
            raise exc_cls(
                exc.status,
                f"[{_request_id}] Error HTTP {exc.status} ({exc.reason}) updating {self._object_type.__name__} "
                f"{gen_obj!r}.",
            )
        if not new_obj:
            raise RuntimeError(
                f"[{_request_id}] Empty response updating {self._object_type.__name__} {gen_obj!r}."
            )
        logger.debug("[%s] Updated %r.", _request_id, new_obj)
        if obj != new_obj:
            logger.warning(
                "[%s] Requested %s to be updated and object returned by server differ. "
                "Requested object:\n%r, returned object:\n%r",
                _request_id,
                self._object_type.__name__,
                obj.dict(),
                new_obj.dict(),
            )
        return new_obj

    async def _request(self, method: str, request_id: str, *args, **kwargs) -> Optional[IDBrokerObject]:
        self.configuration.access_token = await self.token.access_token
        if "_request_timeout" not in kwargs:
            kwargs["_request_timeout"] = ID_BROKER_CLIENT_DEFAULT_TIMEOUT
        async with ApiClient(self.configuration) as api_client:
            api_client.default_headers["Access-Control-Expose-Headers"] = "X-Request-ID"
            api_client.default_headers["X-Request-ID"] = request_id
            api_instance = self._gen_api_handler(api_client)
            meth = getattr(api_instance, self.API_METHODS[method])
            res = await meth(*args, **kwargs)
        if res:
            gen_obj = cast(GenApiObject, res)
            obj = self._object_type.from_gen_obj(gen_obj)
            obj = cast(IDBrokerObject, obj)
            return obj  # GET, POST, PUT
        return res  # DELETE: None


class IDBrokerUser(ProvisioningAPIClient):
    API_METHODS = {
        "delete": "delete_ucsschool_apis_provisioning_v1_school_authority_users_id_delete",
        "get": "get_ucsschool_apis_provisioning_v1_school_authority_users_id_get",
        "head": "get_ucsschool_apis_provisioning_v1_school_authority_users_id_head",
        "post": "post_ucsschool_apis_provisioning_v1_school_authority_users_post",
        "put": "put_ucsschool_apis_provisioning_v1_school_authority_users_id_put",
    }
    _object_type = User
    _gen_api_handler = GenUsersApi

    async def create(self, user: User) -> User:
        """Create user. Returned value is the data from the server."""
        res = await super()._create(
            obj_arg_name="user",
            school_authority=self.school_authority_name,
            user=self._sorted_context_lists(user),
        )
        return cast(User, res)

    async def delete(self, obj_id: str) -> None:
        """Delete user with ID `obj_id`."""
        await self._delete(obj_id=obj_id, school_authority=self.school_authority_name)

    async def exists(self, obj_id: str) -> bool:
        """Check if the user with the ID `obj_id` exists on the server."""
        return await self._exists(obj_id=obj_id, school_authority=self.school_authority_name)

    async def get(self, obj_id: str) -> User:
        """Retrieve user with ID `obj_id` from the server."""
        res = await super()._get(obj_id=obj_id, school_authority=self.school_authority_name)
        return self._sorted_context_lists(cast(User, res))

    async def update(self, user: User) -> User:
        """Modify the user with the ID `user.id` on the server."""
        res = await super()._update(
            obj_arg_name="user",
            school_authority=self.school_authority_name,
            id=user.id,
            user=self._sorted_context_lists(user),
        )
        return cast(User, res)

    @staticmethod
    def _sorted_context_lists(user: User) -> User:
        for context in user.context.values():
            context.roles.sort()
            context.classes.sort()
            context.workgroups.sort()
        return user


class IDBrokerSchool(ProvisioningAPIClient):
    API_METHODS = {
        "get": "get_ucsschool_apis_provisioning_v1_school_authority_schools_id_get",
        "head": "head_ucsschool_apis_provisioning_v1_school_authority_schools_id_head",
        "post": "post_ucsschool_apis_provisioning_v1_school_authority_schools_post",
        "delete": "delete_ucsschool_apis_provisioning_v1_school_authority_schools_id_delete",
    }
    _object_type = School
    _gen_api_handler = GenSchoolsApi

    async def create(self, school: School) -> School:
        """Create school. Returned value is the data from the server."""
        res = await super()._create(
            obj_arg_name="school",
            school_authority=self.school_authority_name,
            school=school,
        )
        return cast(School, res)

    async def exists(self, obj_id: str) -> bool:
        """Check if school with ID `obj_id` exists on the server."""
        return await self._exists(obj_id=obj_id, school_authority=self.school_authority_name)

    async def get(self, obj_id: str) -> School:
        """Retrieve school with ID `obj_id` from the server."""
        res = await super()._get(obj_id=obj_id, school_authority=self.school_authority_name)
        return cast(School, res)

    async def delete(self, obj_id: str) -> None:
        """Delete school with ID `obj_id` from the server."""
        await super()._delete(obj_id=obj_id, school_authority=self.school_authority_name)


class IDBrokerSchoolClass(ProvisioningAPIClient):
    API_METHODS = {
        "get": "get_ucsschool_apis_provisioning_v1_school_authority_classes_id_get",
        "head": "get_ucsschool_apis_provisioning_v1_school_authority_classes_id_head",
        "post": "post_ucsschool_apis_provisioning_v1_school_authority_classes_post",
        "put": "put_ucsschool_apis_provisioning_v1_school_authority_classes_id_put",
        "delete": "delete_ucsschool_apis_provisioning_v1_school_authority_classes_id_delete",
    }
    _object_type = SchoolClass
    _gen_api_handler = GenSchoolClassesApi

    async def create(self, school_class: SchoolClass) -> SchoolClass:
        """Create school class. Returned value is the data from the server."""
        res = await super()._create(
            obj_arg_name="school_class",
            school_authority=self.school_authority_name,
            school_class=self._sorted_members(school_class),
        )
        return cast(SchoolClass, res)

    async def exists(self, obj_id: str) -> bool:
        """Check if the school class with the ID `obj_id` exists on the server."""
        return await self._exists(obj_id=obj_id, school_authority=self.school_authority_name)

    async def get(self, obj_id: str) -> SchoolClass:
        """Retrieve school class with ID `obj_id` from the server."""
        res = await super()._get(obj_id=obj_id, school_authority=self.school_authority_name)
        return self._sorted_members(cast(SchoolClass, res))

    async def update(self, school_class: SchoolClass) -> SchoolClass:
        """Modify the school class with the ID `school_class.id` on the server."""
        res = await super()._update(
            obj_arg_name="school_class",
            school_authority=self.school_authority_name,
            id=school_class.id,
            school_class=self._sorted_members(school_class),
        )
        return cast(SchoolClass, res)

    async def delete(self, obj_id: str) -> None:
        """Delete school_class with ID `obj_id`."""
        await self._delete(obj_id=obj_id, school_authority=self.school_authority_name)

    @staticmethod
    def _sorted_members(school_class: SchoolClass) -> SchoolClass:
        school_class.members.sort()
        return school_class


class IDBrokerWorkGroup(ProvisioningAPIClient):
    API_METHODS = {
        "get": "get_ucsschool_apis_provisioning_v1_school_authority_workgroups_id_get",
        "head": "get_ucsschool_apis_provisioning_v1_school_authority_workgroups_id_head",
        "post": "post_ucsschool_apis_provisioning_v1_school_authority_workgroups_post",
        "put": "put_ucsschool_apis_provisioning_v1_school_authority_workgroups_id_put",
        "delete": "delete_ucsschool_apis_provisioning_v1_school_authority_workgroups_id_delete",
    }
    _object_type = WorkGroup
    _gen_api_handler = GenWorkGroupsApi

    async def create(self, work_group: WorkGroup) -> WorkGroup:
        """Create work group. Returned value is the data from the server."""
        res = await super()._create(
            obj_arg_name="work_group",
            school_authority=self.school_authority_name,
            work_group=self._sorted_members(work_group),
        )
        return cast(WorkGroup, res)

    async def exists(self, obj_id: str) -> bool:
        """Check if the work group with the ID `obj_id` exists on the server."""
        return await self._exists(obj_id=obj_id, school_authority=self.school_authority_name)

    async def get(self, obj_id: str) -> WorkGroup:
        """Retrieve work group with ID `obj_id` from the server."""
        res = await super()._get(obj_id=obj_id, school_authority=self.school_authority_name)
        return self._sorted_members(cast(WorkGroup, res))

    async def update(self, work_group: WorkGroup) -> WorkGroup:
        """Modify the work group with the ID `work_group.id` on the server."""
        res = await super()._update(
            obj_arg_name="work_group",
            school_authority=self.school_authority_name,
            id=work_group.id,
            work_group=self._sorted_members(work_group),
        )
        return cast(WorkGroup, res)

    async def delete(self, obj_id: str) -> None:
        """Delete work_group with ID `obj_id`."""
        await self._delete(obj_id=obj_id, school_authority=self.school_authority_name)

    @staticmethod
    def _sorted_members(work_group: WorkGroup) -> WorkGroup:
        work_group.members.sort()
        return work_group
