#!/usr/bin/env python3
"""
Script to check the ucsschoolRole consistency of user objects in the ID Broker system.
"""
#
# Copyright 2024 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.
#

import sys
from argparse import ArgumentParser, FileType, RawDescriptionHelpFormatter

try:
    from ldap3 import ALL, Connection, Server
except ModuleNotFoundError:
    print("Please install the ldap3 python module to use this script.")
    sys.exit(1)


def check(host, binddn, bindpwd, basedn):
    server = Server(host, get_info=ALL)
    conn = Connection(
        server,
        binddn,
        bindpwd,
        auto_bind=True,
    )

    users = conn.extend.standard.paged_search(
        basedn,
        "(&(objectClass=ucsschoolType)(objectClass=posixAccount))",
        paged_size=100,
        attributes=["ucsschoolRole", "ucsschoolSchool", "modifyTimestamp", "createTimestamp"],
    )
    counter = 0
    corrupt_object_counter = 0
    for user in users:
        counter = counter + 1
        dn = user["dn"]
        try:
            roles = user["attributes"]["ucsschoolRole"]
            schools = set(user["attributes"]["ucsschoolSchool"])
            modify_timestamp = user["attributes"]["modifyTimestamp"]
            create_timestamp = user["attributes"]["createTimestamp"]
            role_parts = [(school, role) for role, _, school in [r.split(":") for r in roles]]
            role_dict = dict()
            for element in role_parts:
                role_dict.setdefault(element[0], set()).add(element[1])
            if schools != set(role_dict.keys()):
                print(
                    f"User {dn} does not have roles for all schools!, "
                    f"created: {create_timestamp} / modified: {modify_timestamp}"
                )
                corrupt_object_counter = corrupt_object_counter + 1
            role_set = list(role_dict.values())[0]
            for roles in role_dict.values():
                if roles != role_set:
                    print(
                        f"User {dn} has mismatching role sets, "
                        f"created: {create_timestamp} / modified: {modify_timestamp}"
                    )
                    corrupt_object_counter = corrupt_object_counter + 1
                    break
        except Exception as exc:
            print(f"Exception during processing of {dn}: {exc}")
        if counter % 1000 == 0:
            print(f"Processed {counter} users.")
    print(f"Processed {counter} users total.")
    print(f"Found {corrupt_object_counter} corrupt objects.")
    if corrupt_object_counter > 0:
        sys.exit(1)
    else:
        sys.exit(0)


def main():
    parser = ArgumentParser(description=__doc__, formatter_class=RawDescriptionHelpFormatter)
    parser.add_argument(
        "--host",
        metavar="HOST",
        help="FQDN of the ldap server to use for checks",
        required=True,
    )
    parser.add_argument(
        "--binddn",
        metavar="DN",
        help="DN of account to use to read from LDAP (e.g. uid=Administrator,cn=users,..)",
        required=True,
    )
    parser.add_argument(
        "--bindpwdfile",
        metavar="FILE",
        type=FileType("r", encoding="UTF-8"),
        help="file containing the password of --binddn",
        required=True,
    )
    parser.add_argument(
        "--basedn",
        metavar="BASEDN",
        help="base dn for to use for checks",
        required=True,
    )
    opt = parser.parse_args()
    password = opt.bindpwdfile.read().strip()
    check(opt.host, opt.binddn, password, opt.basedn)


if __name__ == "__main__":
    main()
