import os
import subprocess
from dataclasses import dataclass
from typing import Generator

import ldap
from playwright.sync_api import TimeoutError as PlaywrightTimeoutError

from ucsschool.lib.models.utils import ucr
from univention.admin.uldap import access

TEST_APP_URL = f"https://kc1.{ucr['domainname']}/univention-test-app/"


def get_ldap_base_from_traeger(hostname: str) -> str:
    ldap_con = ldap.initialize(f"ldap://{hostname}:7389")
    ncs = ldap_con.get_naming_contexts()
    ncs.remove(b"cn=translog")
    return ncs[0].decode("UTF-8")


def get_traeger_ldap_con(traeger: str) -> access:
    """
    UCS_ENV_TRAEGER1_DOMAIN and UCS_ENV_TRAEGER2_DOMAIN
    needs to be set by the (jenkins) job
    dito for the /var/lib/ucs/traeger1[2].secret files
    """
    traegers = {
        "Traeger1": f"ucs-sso.{os.environ.get('UCS_ENV_TRAEGER1_DOMAIN')}",
        "Traeger2": f"ucs-sso.{os.environ.get('UCS_ENV_TRAEGER2_DOMAIN')}",
    }
    hostname = traegers[traeger]
    password = open(f"/var/lib/ucs-test/{traeger}.secret").read().strip()
    ldap_base = get_ldap_base_from_traeger(hostname)
    lo = access(
        hostname,
        start_tls=1,
        base=ldap_base,
        binddn=f"uid=Administrator,cn=users,{ldap_base}",
        bindpw=password,
    )
    return lo


def get_traeger_ldap_admin_con(traeger: str) -> access:
    """
    UCS_ENV_TRAEGER1_DOMAIN and UCS_ENV_TRAEGER2_DOMAIN
    needs to be set by the (jenkins) job
    dito for the /var/lib/ucs/traeger1[2].secret files
    """
    traegers = {
        "Traeger1": f"ucs-sso.{os.environ.get('UCS_ENV_TRAEGER1_DOMAIN')}",
        "Traeger2": f"ucs-sso.{os.environ.get('UCS_ENV_TRAEGER2_DOMAIN')}",
    }
    hostname = traegers[traeger]
    password = (
        subprocess.check_output(
            [
                "univention-ssh",
                f"/var/lib/ucs-test/{traeger}.secret",
                f"Administrator@{hostname}",
                "cat",
                "/etc/ldap.secret",
            ]
        )
        .decode()
        .strip()
    )
    ldap_base = get_ldap_base_from_traeger(hostname)
    lo = access(
        hostname,
        start_tls=1,
        base=ldap_base,
        binddn=f"cn=admin,{ldap_base}",
        bindpw=password,
    )
    return lo


@dataclass
class User:
    username: str
    firstname: str
    lastname: str
    displayName: str
    ucsschoolRole: list
    dn: str
    entryUUID: str


def get_user_from_traeger(traeger: str, filter_l: str) -> Generator[User, None, None]:
    for dn, attrs in get_traeger_ldap_con(traeger).search(filter_l, attr=["*", "+"]):
        yield User(
            username=attrs.get("uid", [b""])[0].decode("utf-8"),
            firstname=attrs.get("givenName", [b""])[0].decode("utf-8"),
            lastname=attrs.get("sn", [b""])[0].decode("utf-8"),
            displayName=attrs.get("gecos", [b""])[0].decode("utf-8"),
            ucsschoolRole=[role.decode("utf-8") for role in attrs.get("ucsschoolRole", [b""])],
            dn=dn,
            entryUUID=attrs.get("entryUUID", [b""])[0].decode("utf-8"),
        )


def load_test_app_page(page, username, password, kdc_idp_hint, login_success=True):
    page.goto(TEST_APP_URL)
    page.get_by_label("From which school authority").click()
    page.get_by_label("From which school authority").fill(kdc_idp_hint)
    page.get_by_role("button", name='"Login"').click()
    page.get_by_text("Login at").wait_for()
    if "simplesamlphp" in page.url:
        page.get_by_label("Username").click()
        page.get_by_label("Username").fill(username)
        page.get_by_label("Username").press("Tab")
    else:
        page.get_by_label("Username or email").click()
        page.get_by_label("Username or email").fill(username)
        page.get_by_label("Username or email").press("Tab")
    page.get_by_label("Password", exact=True).fill(password)
    if "simplesamlphp" in page.url:
        page.get_by_role("button", name="Login").click()
    else:
        page.get_by_role("button", name="Sign In").click()
    try:
        if login_success:
            page.get_by_text("Pseudonym:").wait_for(timeout=10 * 1000)
        else:
            page.get_by_text("Invalid username or password.").wait_for(timeout=10 * 1000)
    except PlaywrightTimeoutError:
        print(page.content())
        raise
