#!/usr/share/ucs-test/runner pytest-3 -s -l -v
## -*- coding: utf-8 -*-
## desc: Check provisioning api generates pseudonyms
## tags: [ucsschool-id-broker]
## packages: [id-broker-common, id-broker-provisioning-api-plugin]
## exposure: dangerous

import random
import uuid
from typing import Dict, Tuple

import pytest
import requests
from six.moves.urllib_parse import urljoin
from utils import _env_is_not_empty

import univention.testing.strings as uts
import univention.testing.ucsschool.ucs_test_school as utu
import univention.testing.udm as udm_test
from ucsschool.lib.models.utils import ucr
from univention.id_broker.pseudonyms import generate_pseudonym
from univention.id_broker.pseudonyms_udm_host import (
    get_service_provider_attribute,
    get_service_provider_secret,
)

pytestmark = pytest.mark.skipif(
    _env_is_not_empty(),
    reason="This test assumes and environment without provisioned users, school_classes and ous.",
)

APP_ID = "ucsschool-apis"
URL_BASE_PATH = "/ucsschool/apis/"
API_ROOT_URL = "http://{}.{}{}".format(ucr["hostname"], ucr["domainname"], URL_BASE_PATH)


@pytest.fixture
def get_res_urls_for_sa():
    def _get_res_urls_for_sa(sa: str) -> Dict[str, str]:
        return {
            "classes": urljoin(API_ROOT_URL, f"provisioning/v1/{sa}/classes/"),
            "schools": urljoin(API_ROOT_URL, f"provisioning/v1/{sa}/schools/"),
            "users": urljoin(API_ROOT_URL, f"provisioning/v1/{sa}/users/"),
        }

    return _get_res_urls_for_sa


def get_access_token(username: str, password: str = "univention") -> str:
    response = requests.post(
        url=f"http://{ucr['hostname']}/ucsschool/apis/auth/token",
        headers={"Content-Type": "application/x-www-form-urlencoded"},
        data=dict(username=username, password=password),
    )
    assert response.status_code == 200, f"{response.__dict__!r}"
    response_json = response.json()
    return response_json["access_token"]


def get_auth_header(username: str) -> Dict[str, str]:
    token = get_access_token(username=username)
    return {"Authorization": f"Bearer {token}"}


@pytest.fixture
def get_token_and_ou_name():
    """
    Creates a user which can modify objects of the school_authority
    and generate a token which can be used to call the provisioning api.
    """

    def _get_token_and_ou_name(
        school_authority: str, create_ou: bool = True
    ) -> Tuple[Dict[str, str], str]:
        provisioning_admin = f"provisioning-{school_authority}"

        if create_ou:
            with utu.UCSTestSchool() as schoolenv:
                ou_name = uts.random_username()
                internal_school_name = f"{school_authority}-{ou_name}"
                schoolenv.create_ou(name_edudc=schoolenv.ucr["hostname"], ou_name=internal_school_name)
                schoolenv.create_teacher(
                    ou_name=internal_school_name,
                    username=provisioning_admin,
                )
                auth_header = get_auth_header(provisioning_admin)
                return auth_header, ou_name
        else:
            with udm_test.UCSTestUDM() as udm:
                udm.create_user(username=provisioning_admin)
                auth_header = get_auth_header(provisioning_admin)
                return auth_header, "NoSchool"

    return _get_token_and_ou_name


def test_school_gets_all_pseudonyms(
    udm, get_test_service_provider, get_token_and_ou_name, get_res_urls_for_sa
):
    sa = uts.random_username()
    service_provider = uts.random_username()
    sa_res_urls = get_res_urls_for_sa(sa)
    auth_header, _ = get_token_and_ou_name(sa, create_ou=False)
    attrs = {
        "id": str(uuid.uuid4()),
        "display_name": uts.random_username(),
        "name": "testou{}".format(random.randint(1000, 9999)),
    }
    service_provider_salt = get_test_service_provider(service_provider)
    response = requests.post(sa_res_urls["schools"], headers=auth_header, json=attrs)
    assert response.status_code == 201
    mod = udm.get("container/ou")
    ou = list(mod.search(f"(ou={sa}-{attrs['name']})"))[0]
    service_attribute = get_service_provider_attribute(service_provider)
    service_provider_salt = get_service_provider_secret(service_provider)
    actual_pseudonym = ou.props.__dict__.get(service_attribute)
    expected_pseudonym = generate_pseudonym(service_provider_salt, attrs["id"], sa)
    assert actual_pseudonym == expected_pseudonym


def test_school_class_gets_all_pseudonyms(
    udm, get_test_service_provider, get_token_and_ou_name, get_res_urls_for_sa
):
    sa = uts.random_username()
    service_provider = uts.random_username()
    sa_res_urls = get_res_urls_for_sa(sa)
    auth_header, ou_name = get_token_and_ou_name(sa)
    attrs = {
        "id": str(uuid.uuid4()),
        "name": "testclass{}".format(random.randint(1000, 9999)),
        "school": ou_name,
        "members": [],
        "description": uts.random_username(),
    }
    service_provider_salt = get_test_service_provider(service_provider)
    response = requests.post(sa_res_urls["classes"], headers=auth_header, json=attrs)
    assert response.status_code == 201
    mod = udm.get("groups/group")
    ou = list(mod.search(f"(cn={sa}-{ou_name}-{attrs['name']})"))[0]
    service_attribute = get_service_provider_attribute(service_provider)
    actual_pseudonym = ou.props.__dict__.get(service_attribute)
    expected_pseudonym = generate_pseudonym(service_provider_salt, attrs["id"], sa)
    assert actual_pseudonym == expected_pseudonym


def test_user_gets_all_pseudonyms(
    udm, get_test_service_provider, get_token_and_ou_name, get_res_urls_for_sa
):
    sa = uts.random_username()
    service_provider = uts.random_username()
    sa_res_urls = get_res_urls_for_sa(sa)
    auth_header, ou_name = get_token_and_ou_name(sa)
    attrs = {
        "id": str(uuid.uuid4()),
        "user_name": uts.random_username(),
        "first_name": uts.random_username(),
        "last_name": uts.random_username(),
        "context": {ou_name: {"classes": [], "roles": ["teacher", "staff"]}},
    }
    service_provider_salt = get_test_service_provider(service_provider)
    response = requests.post(sa_res_urls["users"], headers=auth_header, json=attrs)
    assert response.status_code == 201
    mod = udm.get("users/user")
    user = list(mod.search(f"(uid={sa}-{attrs['user_name']})"))[0]
    service_attribute = get_service_provider_attribute(service_provider)
    actual_pseudonym = user.props.__dict__.get(service_attribute)
    expected_pseudonym = generate_pseudonym(service_provider_salt, attrs["id"], sa)
    assert actual_pseudonym == expected_pseudonym
