import logging

from fastapi import APIRouter, Depends, Path

from self_disclosure_plugin.dependencies import (
    extract_members,
    get_connected_user,
    get_group_by_pseudonym,
    service_provider_from_token,
    token_dep,
)
from self_disclosure_plugin.models import GroupMembers, NoStarStr
from self_disclosure_plugin.sddb_client import SddbGroup, SddbUser
from ucsschool.apis.opa import OPAClient, opa_instance
from ucsschool.apis.utils import get_logger

router = APIRouter(tags=["groups"], dependencies=[Depends(token_dep)])


@router.get("/{id}/users", response_model=GroupMembers)
async def group_users(
    group_pseudonym: NoStarStr = Path(
        ...,
        alias="id",
        description="Pseudonym of group for which to retrieve data.",
        title="Pseudonym",
    ),
    service_provider: str = Depends(service_provider_from_token),
    connected_user: SddbUser = Depends(get_connected_user),
    policy_instance: OPAClient = Depends(opa_instance),
    logger: logging.Logger = Depends(get_logger),
) -> GroupMembers:
    group: SddbGroup = get_group_by_pseudonym(
        pseudonym=group_pseudonym, service_provider=service_provider
    )
    logger.debug(
        "group_pseudonym=%r service_provider=%r connected_user=%r group=%r",
        group_pseudonym,
        service_provider,
        connected_user,
        group,
    )
    await policy_instance.check_policy_true_or_raise(
        "self_disclosure_plugin/groups_members",
        {"user": connected_user.data, "group": group.data},
    )
    return extract_members(group=group, service_provider=service_provider)
