from fastapi import APIRouter, Depends, HTTPException, Path, Request, status
from pydantic import ValidationError
from starlette.responses import Response

from id_broker_common.kelvin import kelvin_session
from id_broker_common.ldap_access import ldap_access, wait_for_replication
from provisioning_plugin.dependencies import (
    check_for_authority_admin,
    get_school_class_by_id,
    member_ids_to_usernames,
)
from provisioning_plugin.models import NonEmptyStr, NoStarStr, SchoolClass
from provisioning_plugin.utils import get_pseudonyms_for_providers
from ucsschool.kelvin.client import InvalidRequest, NoObject, SchoolClass as KelvinSchoolClass, Session

router = APIRouter(tags=["school_classes"], dependencies=[Depends(check_for_authority_admin)])


@router.head("/classes/{id}", response_model=None)
@router.get("/classes/{id}", response_model=SchoolClass)
async def get(
    request: Request,
    class_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    school_class: KelvinSchoolClass = Depends(get_school_class_by_id),
    session: Session = Depends(kelvin_session),
) -> SchoolClass:
    try:
        prov_class = await SchoolClass.from_kelvin_school_class(
            school_class.as_dict(), session=session, fill_members=request.method == "GET"
        )
        return SchoolClass.without_prefix(prov_class, school_authority)
    except (ValidationError, ValueError, NoObject) as exc:
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"The school class {class_id!r} in school authority {school_authority!r} is "
            f"malformed.",
        ) from exc


@router.post("/classes", response_model=SchoolClass, status_code=201)
async def post(
    school_class_data: SchoolClass,
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    session: Session = Depends(kelvin_session),
) -> SchoolClass:
    school_class_data = SchoolClass.with_prefix(school_class_data, school_authority)
    try:
        usernames = await member_ids_to_usernames(
            members=school_class_data.members, school_authority=school_authority
        )
    except HTTPException as exc:
        if exc.status_code == status.HTTP_404_NOT_FOUND:
            raise HTTPException(
                status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
                detail=f"The school class {school_class_data.name!r} in school {school_class_data.school!r} "
                f"in school authority {school_authority!r} could not be created: {exc.detail!r}",
            )
        raise
    school_class = KelvinSchoolClass(
        name=school_class_data.name,
        school=school_class_data.school,
        users=usernames,
        description=school_class_data.description,
        create_share=False,
        udm_properties={
            "ucsschoolRecordUID": school_class_data.id,
            "ucsschoolSourceUID": school_authority,
        },
        session=session,
    )
    async for udm_prop, pseudonym in get_pseudonyms_for_providers(
        school_class_data.id, school_authority
    ):
        school_class.udm_properties[udm_prop] = pseudonym
    try:
        await school_class.save()
    except InvalidRequest as exc:
        if exc.status == status.HTTP_409_CONFLICT:
            await school_class.reload()
            raise HTTPException(
                status_code=exc.status,
                detail={
                    "msg": exc.reason,
                    "conflict_id": school_class.udm_properties["ucsschoolRecordUID"],
                },
            ) from exc
        else:
            raise HTTPException(status_code=exc.status, detail=exc.reason) from exc
    await wait_for_replication(ldap_access(), school_class.dn)
    # To avoid retrieving all members again, at this point we trust that the data was saved correctly
    # It would be safer to use the data from the response. However, that would have a big performance
    # impact. So we'll leave this as it is.
    return SchoolClass.without_prefix(school_class_data, school_authority)


@router.put("/classes/{id}", response_model=SchoolClass)
async def put(
    school_class_data: SchoolClass,
    class_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    school_class: KelvinSchoolClass = Depends(get_school_class_by_id),
    session: Session = Depends(kelvin_session),
):
    school_class_data = SchoolClass.with_prefix(school_class_data, school_authority)
    if school_class_data.school != school_class.school:
        raise HTTPException(
            status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
            detail="The school must not be changed after creation.",
        )
    school_class.name = school_class_data.name
    school_class.description = school_class_data.description
    try:
        usernames = await member_ids_to_usernames(
            members=school_class_data.members, school_authority=school_authority
        )
    except HTTPException as exc:
        if exc.status_code == 404:
            raise HTTPException(
                status_code=status.HTTP_422_UNPROCESSABLE_ENTITY,
                detail=f"The school class {school_class_data.name!r} in school {school_class_data.school!r} "
                f"in school authority {school_authority!r} could not be modified: {exc.detail!r}",
            )
        raise
    school_class.users = usernames
    try:
        await school_class.save()
    except InvalidRequest as exc:
        raise HTTPException(status_code=exc.status, detail=exc.reason) from exc
    return SchoolClass.without_prefix(school_class_data, school_authority)


@router.delete("/classes/{id}", status_code=status.HTTP_204_NO_CONTENT)
async def delete(
    class_id: NoStarStr = Path(
        ...,
        alias="id",
        description="Unique ID of LDAP object on school authority side.",
        title="Object ID",
    ),
    school_authority: NonEmptyStr = Path(
        ...,
        description="Identifier of the school authority this object originates from.",
        title="School authority ID",
    ),
    school_class: KelvinSchoolClass = Depends(get_school_class_by_id),
) -> Response:
    await school_class.delete()
    await wait_for_replication(ldap_access(), school_class.dn, should_exist=False)
    return Response(status_code=status.HTTP_204_NO_CONTENT)
