import os
import random
import re
from html import unescape
from typing import Dict, List

import requests
from locust import HttpUser
from requests.cookies import RequestsCookieJar

SCHOOL_AUTHORITIES_TRAEGER_MAPPING = {
    "Traeger1": {
        "fqdn": os.environ["UCS_ENV_TRAEGER1_FQDN"],
        "idp_hint": os.environ["UCS_ENV_TRAEGER1_IDP_HINT"],
    },
    "Traeger2": {
        "fqdn": os.environ["UCS_ENV_TRAEGER2_FQDN"],
        "idp_hint": os.environ["UCS_ENV_TRAEGER2_IDP_HINT"],
    },
}


def get_saml_cookie(username: str, password: str, fqdn: str) -> RequestsCookieJar:
    with requests.Session() as session:
        url = f"https://{fqdn}/univention/saml/"
        response = session.get(url, verify=False)
        auth_state = re.search(r'name="AuthState" value="([^"]+)"', response.text)
        if auth_state is None:
            raise Exception("no AuthState")
        auth_state = auth_state.groups()[0]
        data = {"username": username, "password": password, "AuthState": auth_state}
        response = session.post(response.url, data=data, allow_redirects=True)
        if "SimpleSAMLAuthToken" not in session.cookies.keys():
            raise Exception("No SimpleSAMLAuthToken")
        return session.cookies


def get_saml_cookies_for_random_users(
    users: Dict[str, Dict[str, Dict[str, str]]], samples: int = 10
) -> List[Dict[str, str]]:
    res = []
    for i in range(samples):
        school_authority = random.choice(list(users.keys()))
        school = random.choice(list(users[school_authority].keys()))
        user = random.choice(list(users[school_authority][school].keys()))
        password = users[school_authority][school][user]
        cookie = get_saml_cookie(
            user, password, SCHOOL_AUTHORITIES_TRAEGER_MAPPING[school_authority]["fqdn"]
        )
        res.append(
            {
                "cookie": cookie,
                "idp_hint": SCHOOL_AUTHORITIES_TRAEGER_MAPPING[school_authority]["idp_hint"],
                "user": user,
                "password": password,
            }
        )
    return res


class TestAppError(Exception):
    ...


class TestApp(HttpUser):
    abstract = True
    endpoint = "univention-test-app/login"

    def extract_saml_relay_state(self, text: str) -> str:
        saml_relay_state = re.search(r'name="RelayState" value="([^"]+)"', text)
        if saml_relay_state is None:
            raise TestAppError("no RelayState")
        return saml_relay_state.groups()[0]

    def extract_saml_auth_state(self, text: str) -> str:
        saml_relay_state = re.search(r'name="AuthState" value="([^"]+)"', text)
        if saml_relay_state is None:
            raise TestAppError("no RelayState")
        return saml_relay_state.groups()[0]

    def extract_auth_redirect_url(self, text: str) -> str:
        redirect_url = re.search(r'action="([^"]+)"', text)
        if redirect_url is None:
            raise TestAppError("no redirect url")
        return redirect_url.groups()[0]

    def extract_saml_request(self, text: str) -> str:
        saml_request = re.search(r'name="SAMLRequest" value="([^"]+)"', text)
        if saml_request is None:
            raise TestAppError("no SAMLRequest")
        return saml_request.groups()[0]

    def extract_saml_response(self, text: str) -> str:
        saml_response = re.search(r'name="SAMLResponse" value="([^"]+)"', text)
        if saml_response is None:
            raise TestAppError("no SAMLResponse")
        return saml_response.groups()[0]

    def extrace_access_token(self, text: str) -> str:
        access_token = re.search(r">{'access_token': '([^']+)'", unescape(text))
        if access_token is None:
            raise TestAppError("no access token")
        return access_token.groups()[0]
