# -*- coding: utf-8 -*-

# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.


import json
from unittest.mock import DEFAULT, AsyncMock, MagicMock, call, patch

import pytest

from idbroker.id_broker_client import IDBrokerWorkGroup as WorkgroupClient, WorkGroup
from idbroker.provisioning_api import ApiException
from idbroker.provisioning_api.models.work_group import WorkGroup as WorkGroupModel


@pytest.mark.asyncio
async def test_workgroup_create(school_authority_conf):
    called_with_kwargs = None

    async def _create(self, **kwargs):
        nonlocal called_with_kwargs
        called_with_kwargs = kwargs

    wg = WorkGroup(id="test", school="TEST-SCHOOL", name="test-group")
    client = WorkgroupClient(school_authority=school_authority_conf, plugin_name="id_broker")
    with patch("idbroker.id_broker_client.ProvisioningAPIClient._create", _create):
        await client.create(wg)
    assert called_with_kwargs == {
        "school_authority": "TEST",
        "work_group": wg,
        "obj_arg_name": "work_group",
    }


@pytest.mark.asyncio
async def test_workgroup_create_conflict(school_authority_conf):
    call_times = 0
    err_resp = MagicMock()
    err_resp.status = 409
    err_resp.reason = "Conflict"
    err_resp.data = json.dumps({"detail": {"msg": "conflict test", "conflict_id": "conflict_workgroup"}})

    async def side_effect(*args, **kwargs):
        nonlocal call_times
        call_times += 1
        if call_times == 1:
            exc = ApiException(http_resp=err_resp)
            raise exc
        return DEFAULT

    wg = WorkGroup(id="test", school="TEST-SCHOOL", name="test-group")

    mock = AsyncMock()
    mock.side_effect = side_effect
    mock.return_value = wg

    client = WorkgroupClient(school_authority=school_authority_conf, plugin_name="id_broker")
    with patch("idbroker.id_broker_client.ProvisioningAPIClient._request", mock):
        await client.create(wg)
    request_id = mock.await_args_list[0][1]["request_id"]
    mock.assert_has_awaits(
        [
            call(
                "post",
                request_id=request_id,
                school_authority="TEST",
                work_group=WorkGroupModel(**wg.dict()),
            ),
            call("delete", id="conflict_workgroup", request_id=request_id, school_authority="TEST"),
            call(
                "post",
                request_id=request_id,
                school_authority="TEST",
                work_group=WorkGroupModel(**wg.dict()),
            ),
        ]
    )


@pytest.mark.asyncio
async def test_workgroup_update(school_authority_conf):
    called_with_kwargs = None

    async def _update(self, **kwargs):
        nonlocal called_with_kwargs
        called_with_kwargs = kwargs

    wg = WorkGroup(id="test", school="TEST-SCHOOL", name="test-group")
    client = WorkgroupClient(school_authority=school_authority_conf, plugin_name="id_broker")
    with patch("idbroker.id_broker_client.ProvisioningAPIClient._update", _update):
        await client.update(wg)
    assert called_with_kwargs == {
        "school_authority": "TEST",
        "work_group": wg,
        "obj_arg_name": "work_group",
        "id": wg.id,
    }


@pytest.mark.asyncio
async def test_workgroup_exists(school_authority_conf):
    called_with_school_authority = None
    called_with_obj_name = None

    async def _exists(self, obj_id: str, school_authority):
        nonlocal called_with_school_authority, called_with_obj_name
        called_with_school_authority = school_authority
        called_with_obj_name = obj_id
        return True

    wg = WorkGroup(id="test-id", school="TEST-SCHOOL", name="test-group")
    client = WorkgroupClient(school_authority=school_authority_conf, plugin_name="id_broker")
    with patch("idbroker.id_broker_client.ProvisioningAPIClient._exists", _exists):
        await client.exists(wg.id)
    assert called_with_school_authority == "TEST"
    assert called_with_obj_name == "test-id"


@pytest.mark.asyncio
async def test_workgroup_get(school_authority_conf):
    called_with_school_authority = None
    called_with_obj_name = None
    members = ["1", "9213", "0", "9", "8", "8"]
    backend_workgroup = WorkGroup(id="test-id", school="TEST-SCHOOL", name="test-group", members=members)

    async def _get(self, obj_id: str, school_authority):
        nonlocal called_with_school_authority, called_with_obj_name
        called_with_school_authority = school_authority
        called_with_obj_name = obj_id
        return backend_workgroup

    client = WorkgroupClient(school_authority=school_authority_conf, plugin_name="id_broker")
    with patch("idbroker.id_broker_client.ProvisioningAPIClient._get", _get):
        res = await client.get(backend_workgroup.id)
    assert res == backend_workgroup
    assert res.members == sorted(members)
    assert called_with_school_authority == "TEST"
    assert called_with_obj_name == "test-id"


@pytest.mark.asyncio
async def test_workgroup_delete(school_authority_conf):
    called_with_school_authority = None
    called_with_obj_name = None

    async def _delete(self, obj_id, school_authority):
        nonlocal called_with_school_authority, called_with_obj_name
        called_with_school_authority = school_authority
        called_with_obj_name = obj_id

    wg = WorkGroup(id="test-id", school="TEST-SCHOOL", name="test-group")
    client = WorkgroupClient(school_authority=school_authority_conf, plugin_name="id_broker")
    with patch("idbroker.id_broker_client.ProvisioningAPIClient._delete", _delete):
        await client.delete(wg.id)

    assert called_with_school_authority == "TEST"
    assert called_with_obj_name == "test-id"
