# -*- coding: utf-8 -*-

# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

import asyncio
import datetime
from typing import Any, Dict

import faker
import pytest
from tenacity import Retrying, retry_if_exception_type, stop_after_attempt, wait_fixed

import ucsschool_id_connector.plugin_loader
from ucsschool.kelvin.client import (
    School as KelvinSchool,
    SchoolClass as KelvinSchoolClass,
    SchoolClassResource as KelvinSchoolClassResource,
    Session,
    User as KelvinUser,
    UserResource,
    WorkGroup as KelvinWorkGroup,
    WorkGroupResource as KelvinWorkGroupResource,
)
from ucsschool_id_connector.ldap_access import LDAPAccess

# load ID Broker plugin
ucsschool_id_connector.plugin_loader.load_plugins()
id_broker = pytest.importorskip("idbroker")
pytestmark = pytest.mark.id_broker

from idbroker.id_broker_client import (  # isort:skip  # noqa: E402
    IDBrokerUser,
    IDBrokerWorkGroup,
)

fake = faker.Faker()
SYNC_TIMEOUT = 60


@pytest.mark.asyncio
async def test_school_exists_after_user_with_non_existing_school_is_synced(
    kelvin_school_on_sender,
    wait_for_kelvin_school_on_id_broker,
    make_sender_user_special,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """
    create kelvin user with non-existing school
    -> school should be created on id broker side
    -> the display_name should be set correct
    """
    school_authority = await set_school_authority_on_id_connector(schools=[kelvin_school_on_sender.name])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    s_a_name = school_authority.name
    sender_user: Dict[str, Any] = await make_sender_user_special(
        ous=[kelvin_school_on_sender.name], s_a_name=s_a_name
    )
    kelvin_school_on_id_broker: KelvinSchool = await wait_for_kelvin_school_on_id_broker(
        s_a_name=s_a_name, name=sender_user["school"]
    )
    assert kelvin_school_on_id_broker.display_name == kelvin_school_on_sender.display_name


@pytest.mark.asyncio
async def test_workgroup_exists_after_user_with_non_existing_workgroup_is_synced(
    kelvin_school_on_sender,
    make_kelvin_workgroup_on_id_connector,
    wait_for_kelvin_workgroup_on_id_broker,
    make_sender_user_special,
    kelvin_session,
    docker_hostname,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """
    Test scenario: The workgroup has not yet been synced.
    To reproduce that (because user.workgroups must exist):

    - create workgroup
    - create user with workgroup
    - delete workgroup on id broker
    - change user on id connector side
    -> workgroup should be created
    """
    school_authority = await set_school_authority_on_id_connector(schools=[kelvin_school_on_sender.name])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    s_a_name = school_authority.name
    sender_kelvin_workgroup: KelvinWorkGroup = await make_kelvin_workgroup_on_id_connector(
        school_name=kelvin_school_on_sender.name, wokrgroup_name="non-existent"
    )
    sender_user = await make_sender_user_special(
        ous=[kelvin_school_on_sender.name],
        s_a_name=s_a_name,
        workgroups={kelvin_school_on_sender.name: ["non-existent"]},
    )
    ldap_access = LDAPAccess()
    results = await ldap_access.search(
        filter_s=f"(&(objectClass=univentionGroup)"
        f"(cn={sender_kelvin_workgroup.school}-{sender_kelvin_workgroup.name}))",
        attributes=["entryUUID"],
    )
    assert len(results) == 1
    sender_group_id = results[0]["entryUUID"].value
    id_broker_group = IDBrokerWorkGroup(school_authority, "id_broker")
    await id_broker_group.delete(sender_group_id)
    assert await id_broker_group.exists(sender_group_id) is False
    sender_user["description"] = "something to trigger"
    updated_sender_user = KelvinUser(session=kelvin_session(docker_hostname), **sender_user)
    updated_sender_user.password = fake.password(
        length=15, special_chars=False
    )  # password must not be used again.
    await updated_sender_user.save()
    await wait_for_kelvin_workgroup_on_id_broker(
        s_a_name=s_a_name,
        school=kelvin_school_on_sender.name,
        name="non-existent",
    )


@pytest.mark.parametrize(
    "school_wildcard", ("none", "full", "partial"), ids=lambda x: f"school_wildcard={x}"
)
@pytest.mark.parametrize(
    "initial_import_mode", ["unset", True, False], ids=lambda x: f"initial_import_mode={x}"
)
@pytest.mark.parametrize("role", ["student", "teacher"])
@pytest.mark.asyncio
async def test_sync_user(
    school_wildcard,
    kelvin_schools_on_sender,
    wait_for_kelvin_user_on_id_broker,
    id_broker_school_auth_conf,
    make_sender_user_special,
    initial_import_mode,
    role,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """
    set initial import mode  -> user should be created without classes
    unset + disable initial import mode -> user should be created with classes
    """
    if initial_import_mode != "unset":
        id_broker_school_auth_conf["plugin_configs"]["id_broker"][
            "initial_import_mode"
        ] = initial_import_mode
    sender_school_1, sender_school_2 = await kelvin_schools_on_sender(2)
    if school_wildcard == "full":
        school_filter = ["*"]
    elif school_wildcard == "none":
        school_filter = [sender_school_1.name, sender_school_2.name]
    elif school_wildcard == "partial":
        school_filter = [
            f"{sender_school_1.name[0]}*{sender_school_1.name[2:]}",
            f"{sender_school_2.name[0]}*{sender_school_2.name[2:]}",
        ]
    else:
        raise ValueError(f"Invalid school_wildcard {school_wildcard}")
    school_authority = await set_school_authority_on_id_connector(schools=school_filter)
    await schedule_schools_delete(
        school_authority=school_authority, schools=[sender_school_1.name, sender_school_2.name]
    )
    s_a_name = school_authority.name
    sender_user: Dict[str, Any] = await make_sender_user_special(
        ous=[sender_school_1.name, sender_school_2.name], s_a_name=s_a_name, roles=(role,)
    )
    remote_user: KelvinUser = await wait_for_kelvin_user_on_id_broker(
        s_a_name=s_a_name, school=sender_school_1.name, name=sender_user["name"]
    )
    assert sorted(f"{s_a_name}-{s}" for s in sender_user["schools"]) == sorted(remote_user.schools)
    if initial_import_mode == 1:
        assert remote_user.school_classes == {}
    else:
        for attempt in Retrying(
            reraise=True,
            stop=stop_after_attempt(3),
            retry=retry_if_exception_type(AssertionError),
            wait=wait_fixed(5),
        ):
            with attempt:
                for key, value in sender_user["school_classes"].items():
                    assert set(value) == set(remote_user.school_classes[f"{s_a_name}-{key}"])


@pytest.mark.parametrize("school_wildcard", ("none", "partial"), ids=lambda x: f"school_wildcard={x}")
@pytest.mark.parametrize(
    "initial_import_mode", ["unset", True, False], ids=lambda x: f"initial_import_mode={x}"
)
@pytest.mark.parametrize("role", ["student", "teacher"])
@pytest.mark.asyncio
async def test_sync_only_user_with_configured_school(
    school_wildcard,
    kelvin_schools_on_sender,
    wait_for_kelvin_user_on_id_broker,
    id_broker_school_auth_conf,
    make_sender_user_special,
    initial_import_mode,
    role,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """
    sender_user1 should never be created
    """
    if initial_import_mode != "unset":
        id_broker_school_auth_conf["plugin_configs"]["id_broker"][
            "initial_import_mode"
        ] = initial_import_mode
    un_configured_sender_school, configured_sender_school = await kelvin_schools_on_sender(2)
    if school_wildcard == "none":
        school_filter = [configured_sender_school.name]
    elif school_wildcard == "partial":
        school_filter = [f"{configured_sender_school.name[0]}*{configured_sender_school.name[2:]}"]
        if (
            f"{configured_sender_school.name[0]}*{configured_sender_school.name[2:]}"
            == f"{un_configured_sender_school.name[0]}*{un_configured_sender_school.name[2:]}"
        ):
            # How probable is this?
            raise ValueError(
                f"{configured_sender_school.name} and {un_configured_sender_school.name} are to similar"
            )
    else:
        raise ValueError(f"Invalid school_wildcard {school_wildcard}")
    school_authority = await set_school_authority_on_id_connector(schools=school_filter)
    await schedule_schools_delete(
        school_authority=school_authority,
        schools=[configured_sender_school.name, un_configured_sender_school.name],
    )
    s_a_name = school_authority.name
    sender_user1: Dict[str, Any] = await make_sender_user_special(
        ous=[un_configured_sender_school.name], s_a_name=s_a_name, roles=(role,)
    )
    sender_user2: Dict[str, Any] = await make_sender_user_special(
        ous=[configured_sender_school.name], s_a_name=s_a_name, roles=(role,)
    )
    remote_user2: KelvinUser = await wait_for_kelvin_user_on_id_broker(
        s_a_name=s_a_name, school=configured_sender_school.name, name=sender_user2["name"]
    )
    assert sorted(f"{s_a_name}-{s}" for s in sender_user2["schools"]) == sorted(remote_user2.schools)
    if initial_import_mode == 1:
        assert remote_user2.school_classes == {}
    else:
        for attempt in Retrying(
            reraise=True,
            stop=stop_after_attempt(3),
            retry=retry_if_exception_type(AssertionError),
            wait=wait_fixed(5),
        ):
            with attempt:
                for key, value in sender_user2["school_classes"].items():
                    assert set(value) == set(remote_user2.school_classes[f"{s_a_name}-{key}"])
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=s_a_name,
        school=un_configured_sender_school.name,
        name=sender_user1["name"],
        must_exist=False,
    )


@pytest.mark.parametrize(
    "initial_import_mode", ["unset", True, False], ids=lambda x: f"initial_import_mode={x}"
)
@pytest.mark.parametrize("role", ["student", "teacher"])
@pytest.mark.asyncio
async def test_remove_user_after_last_configured_school_is_removed(
    kelvin_schools_on_sender,
    wait_for_kelvin_user_on_id_broker,
    id_broker_school_auth_conf,
    make_sender_user_special,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    initial_import_mode,
    role,
    kelvin_session_kwargs,
    docker_hostname,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """
    user should be removed if the last configured school gets removed
    """
    if initial_import_mode != "unset":
        id_broker_school_auth_conf["plugin_configs"]["id_broker"][
            "initial_import_mode"
        ] = initial_import_mode
    un_configured_sender_school, configured_sender_school = await kelvin_schools_on_sender(2)
    school_authority = await set_school_authority_on_id_connector(
        schools=[configured_sender_school.name]
    )
    await schedule_schools_delete(
        school_authority=school_authority, schools=[configured_sender_school.name]
    )
    s_a_name = school_authority.name
    sender_kelvin_school_un_conf_class: KelvinWorkGroup = await make_kelvin_school_class_on_id_connector(
        school_name=un_configured_sender_school.name
    )
    sender_kelvin_school_conf_class: KelvinWorkGroup = await make_kelvin_school_class_on_id_connector(
        school_name=configured_sender_school.name
    )
    sender_kelvin_un_conf_workgroup: KelvinWorkGroup = await make_kelvin_workgroup_on_id_connector(
        school_name=un_configured_sender_school.name
    )
    sender_kelvin_conf_workgroup: KelvinWorkGroup = await make_kelvin_workgroup_on_id_connector(
        school_name=configured_sender_school.name
    )
    sender_user: Dict[str, Any] = await make_sender_user_special(
        ous=[un_configured_sender_school.name, configured_sender_school.name],
        s_a_name=s_a_name,
        roles=(role,),
        workgroups={
            un_configured_sender_school.name: [sender_kelvin_un_conf_workgroup.name],
            configured_sender_school.name: [sender_kelvin_conf_workgroup.name],
        },
        school_classes={
            un_configured_sender_school.name: [sender_kelvin_school_un_conf_class.name],
            configured_sender_school.name: [sender_kelvin_school_conf_class.name],
        },
    )
    remote_user: KelvinUser = await wait_for_kelvin_user_on_id_broker(
        s_a_name=s_a_name, school=un_configured_sender_school.name, name=sender_user["name"]
    )
    if initial_import_mode == 1:
        assert remote_user.school_classes == {}
        assert remote_user.workgroups == {}
    else:
        assert set(sender_user["school_classes"][f"{configured_sender_school.name}"]) == set(
            remote_user.school_classes[f"{s_a_name}-{configured_sender_school.name}"]
        )
        assert set(remote_user.school_classes.keys()) == {f"{s_a_name}-{configured_sender_school.name}"}
        assert set(sender_user["workgroups"][f"{configured_sender_school.name}"]) == set(
            remote_user.workgroups[f"{s_a_name}-{configured_sender_school.name}"]
        )
        assert set(remote_user.workgroups.keys()) == {f"{s_a_name}-{configured_sender_school.name}"}
    assert remote_user.schools == [f"{s_a_name}-{configured_sender_school.name}"]
    assert remote_user.ucsschool_roles == [f"{role}:school:{s_a_name}-{configured_sender_school.name}"]
    async with Session(**kelvin_session_kwargs(docker_hostname)) as session:
        user = await UserResource(session=session).get(name=sender_user["name"])
        user.schools = [un_configured_sender_school.name]
        del user.school_classes[configured_sender_school.name]
        del user.workgroups[configured_sender_school.name]
        await user.save()
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=s_a_name,
        school=configured_sender_school.name,
        name=sender_user["name"],
        must_exist=False,
    )


# IDBrokerPerSAGroupDispatcher


@pytest.mark.parametrize(
    "initial_import_mode", ["unset", True, False], ids=lambda x: f"initial_import_mode={x}"
)
@pytest.mark.asyncio
async def test_sync_school_class(
    kelvin_school_on_sender,
    wait_for_kelvin_school_class_on_id_broker,
    id_broker_school_auth_conf,
    make_kelvin_school_class_on_id_connector,
    initial_import_mode,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """
    set initial import mode -> should not create school_class
    unset + disable initial import mode -> should create school_class
    """
    if initial_import_mode != "unset":
        id_broker_school_auth_conf["plugin_configs"]["id_broker"][
            "initial_import_mode"
        ] = initial_import_mode
    school_authority = await set_school_authority_on_id_connector(schools=[kelvin_school_on_sender.name])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    s_a_name = school_authority.name
    sender_kelvin_school_class: KelvinSchoolClass = await make_kelvin_school_class_on_id_connector(
        school_name=kelvin_school_on_sender.name,
    )
    try:
        await wait_for_kelvin_school_class_on_id_broker(
            s_a_name=s_a_name,
            school=kelvin_school_on_sender.name,
            name=sender_kelvin_school_class.name,
        )
    except AssertionError:
        if initial_import_mode != 1:
            raise AssertionError(
                f"Class should have been created in normal mode (initial_import_mode={initial_import_mode})"
            )
    # in initial_import_mode the class should not be created


@pytest.mark.parametrize(
    "initial_import_mode", ["unset", True, False], ids=lambda x: f"initial_import_mode={x}"
)
@pytest.mark.asyncio
async def test_sync_workgroup(
    kelvin_school_on_sender,
    wait_for_kelvin_workgroup_on_id_broker,
    id_broker_school_auth_conf,
    make_kelvin_workgroup_on_id_connector,
    initial_import_mode,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """
    set initial import mode -> should not create workgroup
    unset + disable initial import mode -> should create workgroup
    """
    if initial_import_mode != "unset":
        id_broker_school_auth_conf["plugin_configs"]["id_broker"][
            "initial_import_mode"
        ] = initial_import_mode
    school_authority = await set_school_authority_on_id_connector(schools=[kelvin_school_on_sender.name])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    s_a_name = school_authority.name
    sender_kelvin_workgroup: KelvinWorkGroup = await make_kelvin_workgroup_on_id_connector(
        school_name=kelvin_school_on_sender.name,
    )
    try:
        await wait_for_kelvin_workgroup_on_id_broker(
            s_a_name=s_a_name,
            school=kelvin_school_on_sender.name,
            name=sender_kelvin_workgroup.name,
        )
    except AssertionError:
        if initial_import_mode != 1:
            raise AssertionError(
                f"Work group should have been created in normal mode (initial_import_mode={initial_import_mode})"
            )
    # in initial_import_mode the work group should not be created


@pytest.mark.asyncio
async def test_remove_workgroup(
    kelvin_school_on_sender,
    wait_for_kelvin_workgroup_on_id_broker,
    make_kelvin_workgroup_on_id_connector,
    kelvin_session_kwargs,
    docker_hostname,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    school_authority = await set_school_authority_on_id_connector(schools=[kelvin_school_on_sender.name])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    s_a_name = school_authority.name
    sender_kelvin_workgroup: KelvinWorkGroup = await make_kelvin_workgroup_on_id_connector(
        school_name=kelvin_school_on_sender.name,
    )
    await wait_for_kelvin_workgroup_on_id_broker(
        s_a_name=s_a_name,
        school=kelvin_school_on_sender.name,
        name=sender_kelvin_workgroup.name,
    )
    async with Session(**kelvin_session_kwargs(docker_hostname)) as session:
        workgroup = await KelvinWorkGroupResource(session=session).get(
            name=sender_kelvin_workgroup.name, school=kelvin_school_on_sender.name
        )
        await workgroup.delete()
    await wait_for_kelvin_workgroup_on_id_broker(
        s_a_name=s_a_name,
        school=kelvin_school_on_sender.name,
        name=sender_kelvin_workgroup.name,
        must_exist=False,
    )


@pytest.mark.asyncio
async def test_remove_school_class(
    kelvin_school_on_sender,
    wait_for_kelvin_school_class_on_id_broker,
    make_kelvin_school_class_on_id_connector,
    kelvin_session_kwargs,
    docker_hostname,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    school_authority = await set_school_authority_on_id_connector(schools=[kelvin_school_on_sender.name])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    s_a_name = school_authority.name
    sender_kelvin_school_class: KelvinWorkGroup = await make_kelvin_school_class_on_id_connector(
        school_name=kelvin_school_on_sender.name,
    )
    await wait_for_kelvin_school_class_on_id_broker(
        s_a_name=s_a_name,
        school=kelvin_school_on_sender.name,
        name=sender_kelvin_school_class.name,
    )
    async with Session(**kelvin_session_kwargs(docker_hostname)) as session:
        school_class = await KelvinSchoolClassResource(session=session).get(
            name=sender_kelvin_school_class.name,
            school=kelvin_school_on_sender.name,
        )
        await school_class.delete()
    await wait_for_kelvin_school_class_on_id_broker(
        s_a_name=s_a_name,
        school=kelvin_school_on_sender.name,
        name=sender_kelvin_school_class.name,
        must_exist=False,
    )


@pytest.mark.asyncio
@pytest.mark.parametrize("group_cls", [KelvinSchoolClass, KelvinWorkGroup])
async def test_school_exists_after_group_with_non_existing_school_is_synced(
    kelvin_school_on_sender,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    wait_for_kelvin_school_on_id_broker,
    group_cls,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """
    create group with non-existing school
    -> school should be created on id broker side
    -> the display_name should be set correct
    """
    if group_cls is KelvinWorkGroup:
        make_group = make_kelvin_workgroup_on_id_connector
    else:
        make_group = make_kelvin_school_class_on_id_connector
    school_authority = await set_school_authority_on_id_connector(schools=[kelvin_school_on_sender.name])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    s_a_name = school_authority.name
    sender_kelvin_group: group_cls = await make_group(
        school_name=kelvin_school_on_sender.name,
    )
    kelvin_school_on_id_broker: KelvinSchool = await wait_for_kelvin_school_on_id_broker(
        s_a_name=s_a_name, name=sender_kelvin_group.school
    )
    assert kelvin_school_on_id_broker.display_name == kelvin_school_on_sender.display_name


@pytest.mark.asyncio
@pytest.mark.parametrize("group_cls", [KelvinSchoolClass, KelvinWorkGroup])
async def test_group_handle_attr_description_empty(
    kelvin_school_on_sender,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    wait_for_kelvin_school_class_on_id_broker,
    wait_for_kelvin_workgroup_on_id_broker,
    group_cls,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """
    Creating a group with an empty description should not
    lead to an error but set it to an empty string on the id broker side.
    """
    if group_cls is KelvinWorkGroup:
        make_group = make_kelvin_workgroup_on_id_connector
        wait_for_group = wait_for_kelvin_workgroup_on_id_broker
    else:
        make_group = make_kelvin_school_class_on_id_connector
        wait_for_group = wait_for_kelvin_school_class_on_id_broker
    school_authority = await set_school_authority_on_id_connector(schools=[kelvin_school_on_sender.name])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    s_a_name = school_authority.name
    sender_group: group_cls = await make_group(
        school_name=kelvin_school_on_sender.name,
    )
    sender_group.description = None
    await sender_group.save()
    kelvin_group_on_id_broker: group_cls = await wait_for_group(
        s_a_name=s_a_name,
        school=kelvin_school_on_sender.name,
        name=sender_group.name,
    )
    assert kelvin_group_on_id_broker.description == sender_group.description


@pytest.mark.asyncio
async def test_handle_attr_users(
    kelvin_school_on_sender,
    make_kelvin_school_class_on_id_connector,
    make_kelvin_workgroup_on_id_connector,
    wait_for_kelvin_workgroup_on_id_broker,
    wait_for_kelvin_school_class_on_id_broker,
    wait_for_kelvin_user_on_id_broker,
    make_sender_user_special,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """
    Tests if the users of the kelvin group are placed inside on the
    id broker side, see  IDBrokerPerSAGroupDispatcher._handle_attr_users

    This also tests implicitly
    - IDBrokerPerSAGroupDispatcher._handle_attr_name
    - IDBrokerPerSAGroupDispatcher._handle_attr_school
    """
    school_authority = await set_school_authority_on_id_connector(schools=[kelvin_school_on_sender.name])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    s_a_name = school_authority.name
    sender_user: Dict[str, Any] = await make_sender_user_special(ous=[kelvin_school_on_sender.name])
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=s_a_name, name=sender_user["name"], school=kelvin_school_on_sender.name
    )
    sender_kelvin_school_class: KelvinSchoolClass = await make_kelvin_school_class_on_id_connector(
        school_name=kelvin_school_on_sender.name,
        users=[sender_user["name"]],
    )
    sender_kelvin_workgroup: KelvinWorkGroup = await make_kelvin_workgroup_on_id_connector(
        school_name=kelvin_school_on_sender.name, users=[sender_user["name"]]
    )
    # The groups are 1st created without users, then the user is created and added to the groups.
    # We'll have to wait with our check until then.
    timeout = datetime.datetime.now() + datetime.timedelta(seconds=SYNC_TIMEOUT)
    while datetime.datetime.now() < timeout:
        used_time = SYNC_TIMEOUT - (timeout - datetime.datetime.now()).seconds
        kelvin_school_class: KelvinSchoolClass = await wait_for_kelvin_school_class_on_id_broker(
            s_a_name=s_a_name,
            school=kelvin_school_on_sender.name,
            name=sender_kelvin_school_class.name,
        )
        kelvin_work_group: KelvinWorkGroup = await wait_for_kelvin_workgroup_on_id_broker(
            s_a_name=s_a_name,
            school=kelvin_school_on_sender.name,
            name=sender_kelvin_workgroup.name,
        )
        try:
            assert kelvin_school_class.users == [f"{s_a_name}-{sender_user['name']}"]
            print(f"OK: After {used_time} sec members as expected.")
            break
        except AssertionError as exc:
            print(f"After {used_time} sec: {exc!r}")
        try:
            assert kelvin_work_group.users == [f"{s_a_name}-{sender_user['name']}"]
            print(f"OK: After {used_time} sec members as expected.")
            break
        except AssertionError as exc:
            print(f"After {used_time} sec: {exc!r}")
        await asyncio.sleep(2)
    else:
        raise AssertionError("Members not as expected.")
    # test IDBrokerPerSAUserDispatcher._handle_attr_context
    remote_user: KelvinUser = await wait_for_kelvin_user_on_id_broker(
        s_a_name=s_a_name, name=sender_user["name"], school=kelvin_school_on_sender.name
    )
    assert (
        sender_kelvin_school_class.name
        in remote_user.school_classes[f"{s_a_name}-{kelvin_school_on_sender.name}"]
    )
    assert (
        sender_kelvin_workgroup.name
        in remote_user.workgroups[f"{s_a_name}-{kelvin_school_on_sender.name}"]
    )
    assert set(sender_user["roles"]) == set(remote_user.roles)


@pytest.mark.asyncio
@pytest.mark.parametrize("group_cls", [KelvinSchoolClass, KelvinWorkGroup])
async def test_group_is_created_with_missing_users(
    kelvin_school_on_sender,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    wait_for_kelvin_user_on_id_broker,
    wait_for_kelvin_workgroup_on_id_broker,
    wait_for_kelvin_school_class_on_id_broker,
    make_sender_user_special,
    group_cls,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """

    Create group with a user existing on the ID Broker and one missing there.
    -> workgroup should be created on ID Broker side
    -> existing user (#2) should be in the group, missing user (#1) is ignored
    """
    if group_cls is KelvinWorkGroup:
        make_group = make_kelvin_workgroup_on_id_connector
        wait_for_group = wait_for_kelvin_workgroup_on_id_broker
    else:
        make_group = make_kelvin_school_class_on_id_connector
        wait_for_group = wait_for_kelvin_school_class_on_id_broker
    school_authority = await set_school_authority_on_id_connector(schools=[kelvin_school_on_sender.name])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    s_a_name = school_authority.name
    sender_user1: Dict[str, Any] = await make_sender_user_special(ous=[kelvin_school_on_sender.name])
    sender_user2: Dict[str, Any] = await make_sender_user_special(ous=[kelvin_school_on_sender.name])
    for sender_user in (sender_user1, sender_user2):
        await wait_for_kelvin_user_on_id_broker(
            s_a_name=s_a_name, name=sender_user["name"], school=kelvin_school_on_sender.name
        )
    ldap_access = LDAPAccess()
    results = await ldap_access.search(
        filter_s=f"(&(objectClass=posixAccount)(uid={sender_user1['name']}))",
        attributes=["entryUUID"],
    )
    assert len(results) == 1
    sender_user1_entry_uuid = results[0]["entryUUID"].value
    id_broker_user = IDBrokerUser(school_authority, "id_broker")
    await id_broker_user.delete(sender_user1_entry_uuid)
    assert not await id_broker_user.exists(sender_user1_entry_uuid)
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=s_a_name,
        name=sender_user1["name"],
        school=kelvin_school_on_sender.name,
        must_exist=False,
    )
    print(
        f"*** User {sender_user1['name']} was deleted on ID Broker system, user {sender_user2['name']} "
        f"exists."
    )
    sender_kelvin_group: group_cls = await make_group(
        school_name=kelvin_school_on_sender.name,
        users=[sender_user1["name"], sender_user2["name"]],
    )
    await sender_kelvin_group.save()
    # _fix_school_class_members() will 1st create the school class without members and then add only the
    # existing users. We'll find the class without members 1st, so give it a bit of time...
    timeout = datetime.datetime.now() + datetime.timedelta(seconds=SYNC_TIMEOUT)
    while datetime.datetime.now() < timeout:
        kelvin_group: group_cls = await wait_for_group(
            s_a_name=s_a_name, name=sender_kelvin_group.name, school=kelvin_school_on_sender.name
        )
        used_time = SYNC_TIMEOUT - (timeout - datetime.datetime.now()).seconds
        try:
            assert kelvin_group.users == [f"{s_a_name}-{sender_user2['name']}"]
            print(f"OK: After {used_time} sec members as expected.")
            break
        except AssertionError as exc:
            print(f"After {used_time} sec: {exc!r}")
        await asyncio.sleep(2)
    else:
        raise AssertionError("Members not as expected.")


@pytest.mark.asyncio
@pytest.mark.parametrize("group_cls", [KelvinSchoolClass, KelvinWorkGroup])
async def test_group_is_updated_with_missing_users(
    kelvin_school_on_sender,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    wait_for_kelvin_user_on_id_broker,
    wait_for_kelvin_workgroup_on_id_broker,
    wait_for_kelvin_school_class_on_id_broker,
    make_sender_user_special,
    group_cls,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """

    Create group with NO users, then add two users, from which only one exists on the ID Broker.
    -> group should be updated on ID Broker side
    -> existing user should be in the group, missing user is ignored
    """
    if group_cls is KelvinWorkGroup:
        make_group = make_kelvin_workgroup_on_id_connector
        wait_for_group = wait_for_kelvin_workgroup_on_id_broker
    else:
        make_group = make_kelvin_school_class_on_id_connector
        wait_for_group = wait_for_kelvin_school_class_on_id_broker
    school_authority = await set_school_authority_on_id_connector(schools=[kelvin_school_on_sender.name])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    s_a_name = school_authority.name
    sender_user1: Dict[str, Any] = await make_sender_user_special(ous=[kelvin_school_on_sender.name])
    sender_user2: Dict[str, Any] = await make_sender_user_special(ous=[kelvin_school_on_sender.name])
    for sender_user in (sender_user1, sender_user2):
        await wait_for_kelvin_user_on_id_broker(
            s_a_name=s_a_name, name=sender_user["name"], school=kelvin_school_on_sender.name
        )
    ldap_access = LDAPAccess()
    results = await ldap_access.search(
        filter_s=f"(&(objectClass=posixAccount)(uid={sender_user1['name']}))",
        attributes=["entryUUID"],
    )
    assert len(results) == 1
    sender_user1_entry_uuid = results[0]["entryUUID"].value
    id_broker_user = IDBrokerUser(school_authority, "id_broker")
    await id_broker_user.delete(sender_user1_entry_uuid)
    assert not await id_broker_user.exists(sender_user1_entry_uuid)
    await wait_for_kelvin_user_on_id_broker(
        s_a_name=s_a_name,
        name=sender_user1["name"],
        school=kelvin_school_on_sender.name,
        must_exist=False,
    )
    print(
        f"*** User {sender_user1['name']} was deleted on ID Broker system, user {sender_user2['name']} "
        f"exists."
    )
    sender_kelvin_group: group_cls = await make_group(
        school_name=kelvin_school_on_sender.name,
        users=[],
    )
    await sender_kelvin_group.save()
    kelvin_group_on_id_broker: group_cls = await wait_for_group(
        s_a_name=s_a_name, name=sender_kelvin_group.name, school=kelvin_school_on_sender.name
    )
    assert kelvin_group_on_id_broker.users == []
    sender_kelvin_group.users.extend([sender_user1["name"], sender_user2["name"]])
    await sender_kelvin_group.save()
    expected = [f"{s_a_name}-{sender_user2['name']}"]
    # If we check immediately, we'll be faster than the ID Connector and will find an empty group.
    timeout = datetime.datetime.now() + datetime.timedelta(seconds=SYNC_TIMEOUT)
    while datetime.datetime.now() < timeout:
        kelvin_group_on_id_broker: group_cls = await wait_for_group(
            s_a_name=s_a_name, name=sender_kelvin_group.name, school=kelvin_school_on_sender.name
        )
        found = kelvin_group_on_id_broker.users
        used_time = SYNC_TIMEOUT - (timeout - datetime.datetime.now()).seconds
        try:
            assert found == expected
            print(f"OK: After {used_time} sec members as expected.")
            break
        except AssertionError as exc:
            print(f"After {used_time} sec: {exc!r}")
        await asyncio.sleep(2)
    else:
        raise AssertionError("Members not as expected.")
