# -*- coding: utf-8 -*-
# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.
import asyncio
import os
from subprocess import check_call
from typing import Any, Dict, List

import pytest

import ucsschool_id_connector.plugin_loader
from ucsschool_id_connector.ldap_access import LDAPAccess

# load ID Broker plugin
ucsschool_id_connector.plugin_loader.load_plugins()
id_broker = pytest.importorskip("idbroker")
pytestmark = pytest.mark.id_broker

import idbroker.initial_sync as InitSync  # isort:skip # noqa: E402
from idbroker.id_broker_client import (  # isort:skip  # noqa: E402
    IDBrokerSchool,
    IDBrokerSchoolClass,
    IDBrokerWorkGroup,
    IDBrokerUser,
    SchoolContext,
    User as IDBrokerUserObject,
)
from conftest import get_class_id, get_workgroup_id  # isort:skip  # noqa: E402

CONFIG_PATH = "/var/lib/univention-appcenter/apps/ucsschool-id-connector/conf/school_authorities"
SYNC_SCRIPT = "/var/lib/univention-appcenter/apps/ucsschool-id-connector/conf/plugins/packages/idbroker/initial_sync.py"  # noqa: E501
ldap = LDAPAccess()


def run_initial_sync_script(call_arguments: List[str] = None):
    if call_arguments is None:
        call_arguments = []

    check_call([SYNC_SCRIPT, "-c", CONFIG_PATH] + call_arguments)


async def get_ou_id(
    ou: str,
) -> str:
    ldap_filter = f"(&(objectClass=ucsschoolOrganizationalUnit)(ou={ou}))"
    objects = await ldap.search(
        base=os.environ.get("ldap_base"), filter_s=ldap_filter, attributes=InitSync.GROUP_ATTRS
    )
    return objects[0].entryUUID.value


async def to_id_broker_user_without_groups(
    user: Dict[str, Any],
) -> IDBrokerUserObject:
    objects = await ldap.search(
        base=os.environ.get("ldap_base"), filter_s=f"(uid={user['name']})", attributes=["entryUUID"]
    )
    u_id = objects[0].entryUUID.value
    return IDBrokerUserObject(
        id=u_id,
        user_name=user["name"],
        first_name=user["firstname"],
        last_name=user["lastname"],
        context={user["school"]: SchoolContext(classes=[], workgroups=[], roles=user["roles"])},
    )


def get_corresponding_broker_obj(sender_obj, id_broker_objects):
    """

    :param sender_obj: kelvin object created on sender
    :param id_broker_objects: id broker objects created from the same type
    :return:
    """
    broker_objects = [obj for obj in id_broker_objects if sender_obj.name == obj.name]
    assert broker_objects
    return broker_objects[0]


@pytest.mark.asyncio
@pytest.fixture
def create_school_on_broker(kelvin_school_on_sender):
    async def _func(school_authority):
        ous = await InitSync.get_ous(ou=kelvin_school_on_sender.name)
        obj = get_corresponding_broker_obj(sender_obj=kelvin_school_on_sender, id_broker_objects=ous)
        school_client = IDBrokerSchool(school_authority, "id_broker")
        await school_client.create(obj)

    return _func


@pytest.mark.asyncio
@pytest.mark.parametrize("dry_run", [True, False], ids=lambda x: f"dry_run={x}")
async def test_sync_object_workgroup(
    create_school_on_broker,
    make_kelvin_workgroup_on_id_connector,
    make_sender_user_special,
    kelvin_school_on_sender,
    dry_run,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    school_authority = await set_school_authority_on_id_connector(schools=["name-is-not-used"])
    await create_school_on_broker(school_authority)
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    InitSync.WORKGROUP_CLIENT = IDBrokerWorkGroup(school_authority, "id_broker")
    work_group = await make_kelvin_workgroup_on_id_connector(school_name=kelvin_school_on_sender.name)
    await make_sender_user_special(
        ous=[kelvin_school_on_sender.name],
        workgroups={kelvin_school_on_sender.name: [work_group.name]},
    )
    work_groups = await InitSync.get_workgroups(ou=kelvin_school_on_sender.name)
    exists = not dry_run
    obj = get_corresponding_broker_obj(sender_obj=work_group, id_broker_objects=work_groups)
    await InitSync.sync_object(obj=obj, sync_members=False, dry_run=dry_run)
    assert await InitSync.WORKGROUP_CLIENT.exists(obj.id) is exists
    if exists:
        group = await InitSync.WORKGROUP_CLIENT.get(obj.id)
        assert group.members == []


@pytest.mark.asyncio
@pytest.mark.parametrize("dry_run", [True, False], ids=lambda x: f"dry_run={x}")
async def test_sync_object_school_class(
    create_school_on_broker,
    make_kelvin_school_class_on_id_connector,
    make_sender_user_special,
    kelvin_school_on_sender,
    dry_run,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    school_authority = await set_school_authority_on_id_connector(schools=["name-is-not-used"])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    InitSync.SCHOOL_CLASS_CLIENT = IDBrokerSchoolClass(school_authority, "id_broker")
    await create_school_on_broker(school_authority)
    user = await make_sender_user_special(
        ous=[kelvin_school_on_sender.name],
    )
    school_class = await make_kelvin_school_class_on_id_connector(
        school_name=kelvin_school_on_sender.name, users=[user["name"]]
    )
    school_classes = await InitSync.get_classes(ou=kelvin_school_on_sender.name)
    exists = not dry_run
    obj = get_corresponding_broker_obj(sender_obj=school_class, id_broker_objects=school_classes)
    await InitSync.sync_object(obj=obj, sync_members=False, dry_run=dry_run)
    assert await InitSync.SCHOOL_CLASS_CLIENT.exists(obj.id) is exists
    if exists:
        group = await InitSync.SCHOOL_CLASS_CLIENT.get(obj.id)
        assert group.members == []


@pytest.mark.asyncio
async def test_sync_object_workgroup_sync_members(
    create_school_on_broker,
    make_kelvin_workgroup_on_id_connector,
    make_sender_user_special,
    kelvin_school_on_sender,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    school_authority = await set_school_authority_on_id_connector(schools=["name-is-not-used"])
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    await create_school_on_broker(school_authority)
    InitSync.WORKGROUP_CLIENT = IDBrokerWorkGroup(school_authority, "id_broker")
    user = await make_sender_user_special(
        ous=[kelvin_school_on_sender.name],
    )
    broker_user = await to_id_broker_user_without_groups(user)
    InitSync.USER_CLIENT = IDBrokerUser(school_authority, "id_broker")
    await InitSync.USER_CLIENT.create(broker_user)
    # add user in the workgroup (on sender)
    work_group = await make_kelvin_workgroup_on_id_connector(
        school_name=kelvin_school_on_sender.name, users=[user["name"]]
    )
    work_groups = await InitSync.get_workgroups(ou=kelvin_school_on_sender.name)
    obj = get_corresponding_broker_obj(sender_obj=work_group, id_broker_objects=work_groups)
    assert await InitSync.WORKGROUP_CLIENT.exists(obj.id) is False
    # This is the line we want to test:
    await InitSync.sync_object(obj=obj, sync_members=True, dry_run=False)
    assert await InitSync.WORKGROUP_CLIENT.exists(obj.id) is True
    group = await InitSync.WORKGROUP_CLIENT.get(obj.id)
    assert group.members == [broker_user.id]


@pytest.mark.asyncio
async def test_sync_object_school_class_sync_members(
    make_sender_user_special,
    kelvin_school_on_sender,
    make_kelvin_school_class_on_id_connector,
    create_school_on_broker,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    school_authority = await set_school_authority_on_id_connector(
        schools=["name-is-not-used"], active=False
    )
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    await create_school_on_broker(school_authority)
    InitSync.USER_CLIENT = IDBrokerUser(school_authority, "id_broker")
    InitSync.SCHOOL_CLASS_CLIENT = IDBrokerSchoolClass(school_authority, "id_broker")
    await asyncio.sleep(3)
    user = await make_sender_user_special(
        ous=[kelvin_school_on_sender.name],
    )
    broker_user = await to_id_broker_user_without_groups(user)
    await InitSync.USER_CLIENT.create(broker_user)
    # add user in the school class (on sender)
    school_class = await make_kelvin_school_class_on_id_connector(
        school_name=kelvin_school_on_sender.name, users=[user["name"]]
    )
    school_classes = await InitSync.get_classes(ou=kelvin_school_on_sender.name)
    obj = get_corresponding_broker_obj(sender_obj=school_class, id_broker_objects=school_classes)
    assert await InitSync.SCHOOL_CLASS_CLIENT.exists(obj.id) is False
    # This is the line we want to test:
    await InitSync.sync_object(obj=obj, sync_members=True, dry_run=False)
    assert await InitSync.SCHOOL_CLASS_CLIENT.exists(obj.id) is True
    group = await InitSync.SCHOOL_CLASS_CLIENT.get(obj.id)
    assert group.members == [broker_user.id]


@pytest.mark.asyncio
@pytest.mark.parametrize("dry_run", [True, False], ids=lambda x: f"dry_run={x}")
async def test_sync_object_school_single(
    kelvin_schools_on_sender, dry_run, schedule_schools_delete, set_school_authority_on_id_connector
):
    school_authority = await set_school_authority_on_id_connector(
        schools=["name-is-not-used"], active=False
    )
    InitSync.SCHOOL_CLIENT = IDBrokerSchool(school_authority, "id_broker")
    await asyncio.sleep(3)
    school = await kelvin_schools_on_sender(1)
    school_name = school[0].name
    await schedule_schools_delete(school_authority=school_authority, schools=[school_name])
    ous = await InitSync.get_ous(ou=school_name)
    exists = not dry_run
    obj = get_corresponding_broker_obj(sender_obj=school[0], id_broker_objects=ous)
    assert await InitSync.SCHOOL_CLIENT.exists(obj.id) is False
    await InitSync.sync_object(obj=obj, sync_members=False, dry_run=dry_run)
    assert await InitSync.SCHOOL_CLIENT.exists(obj.id) is exists


@pytest.mark.asyncio
async def test_script_dry_run(
    kelvin_school_on_sender,
    make_sender_user_special,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """Scenario
    The test users & groups are created, nothing should be synced."""
    school_authority = await set_school_authority_on_id_connector(
        schools=["name-is-not-used"], active=False
    )
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    school_client = IDBrokerSchool(school_authority, "id_broker")
    classes_client = IDBrokerSchoolClass(school_authority, "id_broker")
    workgroups_client = IDBrokerWorkGroup(school_authority, "id_broker")
    await asyncio.sleep(3)
    work_group = await make_kelvin_workgroup_on_id_connector(school_name=kelvin_school_on_sender.name)
    user = await make_sender_user_special(
        ous=[kelvin_school_on_sender.name],
        workgroups={kelvin_school_on_sender.name: [work_group.name]},
    )
    school_class = await make_kelvin_school_class_on_id_connector(
        school_name=kelvin_school_on_sender.name, users=[user["name"]]
    )
    school_class_id = await get_class_id(ou=kelvin_school_on_sender.name, name=school_class.name)
    work_group_id = await get_workgroup_id(ou=kelvin_school_on_sender.name, name=work_group.name)
    ou_id = await get_ou_id(ou=kelvin_school_on_sender.name)
    assert await classes_client.exists(school_class_id) is False
    assert await workgroups_client.exists(work_group_id) is False
    assert await school_client.exists(ou_id) is False
    run_initial_sync_script(call_arguments=["--dry-mode"])
    assert await classes_client.exists(school_class_id) is False
    assert await workgroups_client.exists(work_group_id) is False
    assert await school_client.exists(ou_id) is False


@pytest.mark.asyncio
async def test_script_single_ou(
    kelvin_school_on_sender,
    kelvin_schools_on_sender,
    make_sender_user_special,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """
    Scenario:
    The test users & groups in one school are created
     -> school and groups without members should be synced.
     -> Second school should not be synced.
    """
    school_authority = await set_school_authority_on_id_connector(
        schools=["name-is-not-used"], active=False
    )
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    school_client = IDBrokerSchool(school_authority, "id_broker")
    classes_client = IDBrokerSchoolClass(school_authority, "id_broker")
    workgroups_client = IDBrokerWorkGroup(school_authority, "id_broker")
    await asyncio.sleep(3)
    work_group = await make_kelvin_workgroup_on_id_connector(school_name=kelvin_school_on_sender.name)
    user = await make_sender_user_special(
        ous=[kelvin_school_on_sender.name],
        workgroups={kelvin_school_on_sender.name: [work_group.name]},
    )
    school_class = await make_kelvin_school_class_on_id_connector(
        school_name=kelvin_school_on_sender.name, users=[user["name"]]
    )
    other_school = await kelvin_schools_on_sender(1)
    other_school_name = other_school[0].name
    other_school_class = await make_kelvin_school_class_on_id_connector(school_name=other_school_name)
    await make_sender_user_special(
        ous=[other_school_name],
        school_classes={other_school_name: [other_school_class.name]},
    )
    run_initial_sync_script(
        call_arguments=["--school", kelvin_school_on_sender.name],
    )
    work_group_id = await get_workgroup_id(ou=kelvin_school_on_sender.name, name=work_group.name)
    school_class_id = await get_class_id(ou=kelvin_school_on_sender.name, name=school_class.name)
    ou_id = await get_ou_id(ou=kelvin_school_on_sender.name)
    assert await school_client.exists(ou_id) is True
    assert await workgroups_client.exists(work_group_id) is True
    assert await classes_client.exists(school_class_id) is True
    group = await workgroups_client.get(work_group_id)
    assert group.members == []
    group = await classes_client.get(school_class_id)
    assert group.members == []
    # other schools should not be synced.
    school_class_id = await get_class_id(ou=other_school_name, name=other_school_class.name)
    ou_id = await get_ou_id(ou=other_school_name)
    assert await school_client.exists(ou_id) is False
    assert await classes_client.exists(school_class_id) is False


@pytest.mark.asyncio
async def test_script_single_ou_sync_members(
    kelvin_school_on_sender,
    make_sender_user_special,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    create_school_on_broker,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """
    Scenario:
    The test users & groups in one school are created
    -> schools & groups without members should be synced.
    """
    school_authority = await set_school_authority_on_id_connector(
        schools=["name-is-not-used"], active=False
    )
    await create_school_on_broker(school_authority)
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    InitSync.WORKGROUP_CLIENT = IDBrokerWorkGroup(school_authority, "id_broker")
    InitSync.SCHOOL_CLASS_CLIENT = IDBrokerSchoolClass(school_authority, "id_broker")
    await asyncio.sleep(3)
    user = await make_sender_user_special(
        ous=[kelvin_school_on_sender.name],
    )
    broker_user = await to_id_broker_user_without_groups(user)
    id_broker_user_client = IDBrokerUser(school_authority, "id_broker")
    await id_broker_user_client.create(broker_user)
    # add user in the workgroup (on sender)
    work_group = await make_kelvin_workgroup_on_id_connector(
        school_name=kelvin_school_on_sender.name, users=[user["name"]]
    )
    # add user in the school class (on sender)
    school_class = await make_kelvin_school_class_on_id_connector(
        school_name=kelvin_school_on_sender.name, users=[user["name"]]
    )
    work_groups = await InitSync.get_workgroups(ou=kelvin_school_on_sender.name)
    wg_obj = get_corresponding_broker_obj(sender_obj=work_group, id_broker_objects=work_groups)
    school_classes = await InitSync.get_classes(ou=kelvin_school_on_sender.name)
    sc_obj = get_corresponding_broker_obj(sender_obj=school_class, id_broker_objects=school_classes)
    assert await InitSync.WORKGROUP_CLIENT.exists(wg_obj.id) is False
    assert await InitSync.SCHOOL_CLASS_CLIENT.exists(sc_obj.id) is False
    # This is the line we want to test:
    run_initial_sync_script(
        call_arguments=["--school", kelvin_school_on_sender.name, "--members"],
    )
    assert await InitSync.WORKGROUP_CLIENT.exists(wg_obj.id) is True
    assert await InitSync.SCHOOL_CLASS_CLIENT.exists(sc_obj.id) is True
    group = await InitSync.WORKGROUP_CLIENT.get(wg_obj.id)
    assert group.members == [broker_user.id]
    group = await InitSync.SCHOOL_CLASS_CLIENT.get(sc_obj.id)
    assert group.members == [broker_user.id]


@pytest.mark.asyncio
async def test_script_only_sync_groups_with_members(
    make_kelvin_school_class_on_id_connector,
    make_kelvin_workgroup_on_id_connector,
    kelvin_school_on_sender,
    schedule_schools_delete,
    set_school_authority_on_id_connector,
):
    """Empty groups should not be synced."""
    school_authority = await set_school_authority_on_id_connector(
        schools=["name-is-not-used"], active=False
    )
    await schedule_schools_delete(
        school_authority=school_authority, schools=[kelvin_school_on_sender.name]
    )
    classes_client = IDBrokerSchoolClass(school_authority, "id_broker")
    workgroups_client = IDBrokerWorkGroup(school_authority, "id_broker")
    await asyncio.sleep(3)
    school_class = await make_kelvin_school_class_on_id_connector(
        school_name=kelvin_school_on_sender.name
    )
    work_group = await make_kelvin_workgroup_on_id_connector(school_name=kelvin_school_on_sender.name)
    work_group_id = await get_workgroup_id(ou=f"{kelvin_school_on_sender.name}", name=work_group.name)
    school_class_id = await get_class_id(ou=f"{kelvin_school_on_sender.name}", name=school_class.name)
    run_initial_sync_script(
        call_arguments=["--school", kelvin_school_on_sender.name],
    )
    assert await classes_client.exists(school_class_id) is False
    assert await workgroups_client.exists(work_group_id) is False


@pytest.mark.asyncio
async def test_script_multi_schools(
    kelvin_schools_on_sender,
    make_sender_user_special,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    set_school_authority_on_id_connector,
):
    """
    Scenario:
    The test users & groups in multiple schools are created and should be synced.
    WARNING: This test can take a very long time if many schools exist on the sender.
    """
    school_authority = await set_school_authority_on_id_connector(
        schools=["name-is-not-used"], active=False
    )
    school_client = IDBrokerSchool(school_authority, "id_broker")
    classes_client = IDBrokerSchoolClass(school_authority, "id_broker")
    workgroups_client = IDBrokerWorkGroup(school_authority, "id_broker")
    await asyncio.sleep(3)
    sender_school_1, sender_school2 = await kelvin_schools_on_sender(2)
    work_group = await make_kelvin_workgroup_on_id_connector(school_name=sender_school_1.name)
    user = await make_sender_user_special(
        ous=[sender_school_1.name],
        workgroups={sender_school_1.name: [work_group.name]},
    )
    school_class = await make_kelvin_school_class_on_id_connector(
        school_name=sender_school_1.name, users=[user["name"]]
    )
    other_school_name = sender_school2.name
    other_work_group = await make_kelvin_workgroup_on_id_connector(school_name=other_school_name)
    other_user = await make_sender_user_special(
        ous=[other_school_name],
        workgroups={other_school_name: [other_work_group.name]},
    )
    other_school_class = await make_kelvin_school_class_on_id_connector(
        school_name=other_school_name, users=[other_user["name"]]
    )
    run_initial_sync_script()
    for ou in (sender_school_1.name, other_school_name):
        ou_id = await get_ou_id(ou=ou)
        assert await school_client.exists(ou_id) is True
    # School classes should be created and have no members.
    school_class_id = await get_class_id(ou=sender_school_1.name, name=school_class.name)
    assert await classes_client.exists(school_class_id) is True
    group = await classes_client.get(school_class_id)
    assert group.members == []
    school_class_id = await get_class_id(ou=sender_school2.name, name=other_school_class.name)
    assert await classes_client.exists(school_class_id) is True
    group = await classes_client.get(school_class_id)
    assert group.members == []
    # Workgroups should be created and have no members.
    work_group_id = await get_workgroup_id(ou=sender_school_1.name, name=work_group.name)
    assert await workgroups_client.exists(work_group_id) is True
    group = await workgroups_client.get(work_group_id)
    assert group.members == []
    work_group_id = await get_workgroup_id(ou=sender_school2.name, name=other_work_group.name)
    assert await workgroups_client.exists(work_group_id) is True
    group = await workgroups_client.get(work_group_id)
    assert group.members == []


@pytest.mark.asyncio
async def test_script_multi_schools_sync_members(
    kelvin_schools_on_sender,
    make_sender_user_special,
    make_kelvin_workgroup_on_id_connector,
    make_kelvin_school_class_on_id_connector,
    set_school_authority_on_id_connector,
):
    """
    Scenario:
    The test users & groups in multiple schools are created and should be synced.
    WARNING: This test can take a very long time if many schools exist on the sender.
    """
    school_authority = await set_school_authority_on_id_connector(
        schools=["name-is-not-used"], active=False
    )
    school_client = IDBrokerSchool(school_authority, "id_broker")
    classes_client = IDBrokerSchoolClass(school_authority, "id_broker")
    workgroups_client = IDBrokerWorkGroup(school_authority, "id_broker")
    id_broker_user_client = IDBrokerUser(school_authority, "id_broker")
    await asyncio.sleep(3)
    # create school on sender & receiver
    sender_school_1, sender_school2 = await kelvin_schools_on_sender(2)
    for school in (sender_school_1, sender_school2):
        ous = await InitSync.get_ous(ou=school.name)
        obj = get_corresponding_broker_obj(sender_obj=school, id_broker_objects=ous)
        await school_client.create(obj)
    # create a user in one class and one workgroup in school 1
    user = await make_sender_user_special(
        ous=[sender_school_1.name],
    )
    work_group = await make_kelvin_workgroup_on_id_connector(
        school_name=sender_school_1.name, users=[user["name"]]
    )
    school_class = await make_kelvin_school_class_on_id_connector(
        school_name=sender_school_1.name, users=[user["name"]]
    )
    broker_user_school_1 = await to_id_broker_user_without_groups(user)
    # create a user in one class and one workgroup in school 2
    school_2_name = sender_school2.name
    other_user = await make_sender_user_special(
        ous=[school_2_name],
    )
    other_work_group = await make_kelvin_workgroup_on_id_connector(
        school_name=school_2_name, users=[other_user["name"]]
    )
    school_class_school2 = await make_kelvin_school_class_on_id_connector(
        school_name=school_2_name, users=[other_user["name"]]
    )
    broker_user_school_2 = await to_id_broker_user_without_groups(other_user)

    await id_broker_user_client.create(broker_user_school_1)
    await id_broker_user_client.create(broker_user_school_2)

    work_groups_school_1 = await InitSync.get_workgroups(ou=sender_school_1.name)
    work_groups_school_2 = await InitSync.get_workgroups(ou=sender_school2.name)
    school_classes_school_1 = await InitSync.get_classes(ou=sender_school_1.name)
    school_classes_school_2 = await InitSync.get_classes(ou=sender_school2.name)

    wg_school_1_obj = get_corresponding_broker_obj(
        sender_obj=work_group, id_broker_objects=work_groups_school_1
    )
    wg_school_2_obj = get_corresponding_broker_obj(
        sender_obj=other_work_group, id_broker_objects=work_groups_school_2
    )
    sc_school_1__obj = get_corresponding_broker_obj(
        sender_obj=school_class, id_broker_objects=school_classes_school_1
    )
    sc_school_2__obj = get_corresponding_broker_obj(
        sender_obj=school_class_school2, id_broker_objects=school_classes_school_2
    )

    assert await workgroups_client.exists(wg_school_1_obj.id) is False
    assert await workgroups_client.exists(wg_school_2_obj.id) is False
    assert await classes_client.exists(sc_school_1__obj.id) is False
    assert await classes_client.exists(sc_school_2__obj.id) is False
    run_initial_sync_script(call_arguments=["--members"])
    # groups should be created and have members.
    assert await workgroups_client.exists(wg_school_1_obj.id) is True
    assert await workgroups_client.exists(wg_school_2_obj.id) is True
    assert await classes_client.exists(sc_school_1__obj.id) is True
    assert await classes_client.exists(sc_school_2__obj.id) is True
    group = await classes_client.get(sc_school_1__obj.id)
    assert group.members == [broker_user_school_1.id]
    group = await classes_client.get(sc_school_2__obj.id)
    assert group.members == [broker_user_school_2.id]
    group = await workgroups_client.get(wg_school_1_obj.id)
    assert group.members == [broker_user_school_1.id]
    group = await workgroups_client.get(wg_school_2_obj.id)
    assert group.members == [broker_user_school_2.id]
