from unittest.mock import AsyncMock

import pytest

import provisioning_plugin.dependencies
from id_broker_common.kelvin import kelvin_session
from provisioning_plugin.models import WorkGroup
from provisioning_plugin.routes.v1 import workgroups
from tests.unittests.utils import (
    CLASS_SIZE,
    DEFAULT_SCHOOL_AUTHORITY,
    PREFIX,
    Dictable,
    get_preconfigured_session_mock,
    kelvin_student,
    kelvin_workgroup,
)
from ucsschool.kelvin.client.exceptions import InvalidRequest


def kelvin_funky_workgroup() -> Dictable:
    class FunkyDictable(Dictable):
        async def save(self, *args, **kwargs):
            from ucsschool.kelvin.client import InvalidRequest

            raise InvalidRequest(msg="test", status=422, reason="test-reason")

    return FunkyDictable(
        name="Wg1",
        description="Wg1 of School1",
        school=f"{PREFIX}School1",
        users=[f"{PREFIX}student{i}" for i in range(CLASS_SIZE)],
        udm_properties={"ucsschoolRecordUID": "Wg1-id"},
    )


@pytest.mark.parametrize("method", ["GET", "HEAD"])
def test_workgroup_by_id(func_client, monkeypatch, method, broker_workgroup):
    exp = WorkGroup.with_prefix(WorkGroup(**broker_workgroup.dict()), DEFAULT_SCHOOL_AUTHORITY)
    monkeypatch.setattr(WorkGroup, "from_kelvin_workgroup", AsyncMock(return_value=exp))
    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.get_workgroup_by_id: kelvin_workgroup,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    query = _client.get if method == "GET" else _client.head
    response = query(url=f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/workgroups/{broker_workgroup.id}")
    assert response.status_code == 200
    if method == "GET":
        res = response.json()
        res["members"] = sorted(res["members"])
        assert res == broker_workgroup
    else:
        assert response.status_code == 200, response.__dict__
        assert response.content == b""


def test_workgroup_by_id_internal_error(func_client, monkeypatch, broker_workgroup):
    async def from_kelvin_workgroup(*args, **kwargs):
        raise ValueError()

    monkeypatch.setattr(WorkGroup, "from_kelvin_workgroup", from_kelvin_workgroup)
    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.get_workgroup_by_id: kelvin_workgroup,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    response = _client.get(url=f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/workgroups/{broker_workgroup.id}")
    assert response.status_code == 500
    assert response.json() == {
        "detail": "The workgroup 'Wg1-id' in school authority 'test_authority' is malformed."
    }


@pytest.mark.parametrize(
    "method,status_code",
    [
        ("POST", 201),
        ("PUT", 200),
    ],
)
def test_put_post_workgroup(
    func_client, monkeypatch, method, status_code, broker_workgroup, async_iterable
):
    monkeypatch.setattr(
        workgroups,
        "member_ids_to_usernames",
        AsyncMock(return_value=[kelvin_student(i) for i in range(CLASS_SIZE)]),
    )
    _client = func_client(
        custom_overrides={
            kelvin_session: get_preconfigured_session_mock(
                {
                    "post.return_value": kelvin_workgroup(),
                }
            ),
            provisioning_plugin.dependencies.get_workgroup_by_id: kelvin_workgroup,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    monkeypatch.setattr(
        workgroups,
        "get_pseudonyms_for_providers",
        async_iterable,
    )
    url = f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/workgroups"
    query = None
    if method == "POST":
        query = _client.post
    elif method == "PUT":
        query = _client.put
        url += f"/{broker_workgroup.id}"
    response = query(
        url,
        json=broker_workgroup,
    )
    assert response.status_code == status_code
    res = response.json()
    res["members"] = sorted(res["members"])
    assert res == broker_workgroup, response.json()


def test_put_invalid_kelvin_request(func_client, monkeypatch, broker_workgroup, async_iterable):
    monkeypatch.setattr(
        workgroups,
        "member_ids_to_usernames",
        AsyncMock(return_value=[kelvin_student(i) for i in range(CLASS_SIZE)]),
    )
    _client = func_client(
        custom_overrides={
            kelvin_session: get_preconfigured_session_mock(),
            provisioning_plugin.dependencies.get_workgroup_by_id: kelvin_funky_workgroup,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    monkeypatch.setattr(
        workgroups,
        "get_pseudonyms_for_providers",
        async_iterable,
    )
    url = f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/workgroups"
    url += f"/{broker_workgroup.id}"
    response = _client.put(
        url,
        json=broker_workgroup,
    )
    assert response.status_code == 422, response.json()
    assert response.json() == {"detail": "test-reason"}


def test_post_invalid_kelvin_request(func_client, monkeypatch, broker_workgroup, async_iterable):
    async def post_mock(*args, **kwargs):
        raise InvalidRequest(msg="test", status=422, reason="test-reason")

    monkeypatch.setattr(
        workgroups,
        "member_ids_to_usernames",
        AsyncMock(return_value=[kelvin_student(i) for i in range(CLASS_SIZE)]),
    )
    _client = func_client(
        custom_overrides={
            kelvin_session: get_preconfigured_session_mock({"post": post_mock}),
            provisioning_plugin.dependencies.get_workgroup_by_id: kelvin_workgroup,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    monkeypatch.setattr(
        workgroups,
        "get_pseudonyms_for_providers",
        async_iterable,
    )
    url = f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/workgroups"
    response = _client.post(
        url,
        json=broker_workgroup,
    )
    assert response.status_code == 422, response.json()
    assert response.json() == {"detail": "test-reason"}


def test_post_conflict_kelvin_request(func_client, monkeypatch, broker_workgroup, async_iterable):
    async def post_mock(*args, **kwargs):
        raise InvalidRequest(msg="test", status=409, reason="test-reason")

    async def reload_mock(*args, **kwargs):
        return

    monkeypatch.setattr(
        workgroups,
        "member_ids_to_usernames",
        AsyncMock(return_value=[kelvin_student(i) for i in range(CLASS_SIZE)]),
    )
    _client = func_client(
        custom_overrides={
            kelvin_session: get_preconfigured_session_mock({"post": post_mock}),
            provisioning_plugin.dependencies.get_workgroup_by_id: kelvin_workgroup,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    monkeypatch.setattr(
        workgroups.KelvinWorkGroup,
        "reload",
        reload_mock,
    )
    monkeypatch.setattr(
        workgroups,
        "get_pseudonyms_for_providers",
        async_iterable,
    )
    url = f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/workgroups"
    response = _client.post(
        url,
        json=broker_workgroup,
    )
    assert response.status_code == 409, response.json()
    assert response.json() == {"detail": {"conflict_id": broker_workgroup["id"], "msg": "test-reason"}}


@pytest.mark.parametrize(
    "method",
    [
        "POST",
        "PUT",
    ],
)
def test_put_post_workgroup_missing_member(
    func_client, monkeypatch, method, broker_workgroup, async_iterable
):
    class ldap_access:
        async def search(self, filter_s, **kwargs):
            return []

    monkeypatch.setattr(
        provisioning_plugin.dependencies,
        "ldap_access",
        ldap_access,
    )

    _client = func_client(
        custom_overrides={
            kelvin_session: get_preconfigured_session_mock(
                {
                    "post.return_value": kelvin_workgroup(),
                }
            ),
            provisioning_plugin.dependencies.get_workgroup_by_id: kelvin_workgroup,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    monkeypatch.setattr(
        workgroups,
        "get_pseudonyms_for_providers",
        async_iterable,
    )
    url = f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/workgroups"
    query = None
    change = "created"
    if method == "POST":
        query = _client.post
    elif method == "PUT":
        query = _client.put
        url += f"/{broker_workgroup.id}"
        change = "modified"
    response = query(
        url,
        json=broker_workgroup,
    )
    assert response.status_code == 422, response.json()
    assert response.json()["detail"] in [
        f"The workgroup '{broker_workgroup['name']}' in school "
        f"'{DEFAULT_SCHOOL_AUTHORITY}-{broker_workgroup['school']}' in school authority "
        f"'{DEFAULT_SCHOOL_AUTHORITY}' could not be {change}: "
        f"\"Object '{member}' in school authority '{DEFAULT_SCHOOL_AUTHORITY}' not found.\""
        for member in broker_workgroup["members"]
    ]


def test_put_school_change_raises_exception(func_client, broker_workgroup, async_iterable):
    broker_workgroup["school"] = "different_school"
    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.get_workgroup_by_id: kelvin_workgroup,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    response = _client.put(
        f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/workgroups/{broker_workgroup.id}",
        json=broker_workgroup,
    )
    assert response.status_code == 422, response.json()
    assert response.json() == {"detail": "The school must not be changed after creation."}


def test_delete_workgroup(func_client, broker_workgroup):
    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.get_workgroup_by_id: kelvin_workgroup,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    url = f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/workgroups/{broker_workgroup.id}"
    response = _client.delete(
        url,
    )
    assert response.status_code == 204, response.json()
