from unittest.mock import AsyncMock

import pytest

import provisioning_plugin.dependencies
from id_broker_common.kelvin import kelvin_session
from provisioning_plugin.routes.v1 import users
from tests.unittests.utils import (
    DEFAULT_SCHOOL_AUTHORITY,
    get_preconfigured_session_mock,
    kelvin_student,
)
from ucsschool.kelvin.client.exceptions import InvalidRequest


@pytest.mark.parametrize("method", ["GET", "HEAD"])
def test_user_by_user_id(func_client, method, broker_student):
    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.get_user_by_id: kelvin_student,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    query = _client.get if method == "GET" else _client.head
    response = query(url=f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/student1")
    assert response.status_code == 200
    if method == "GET":
        assert response.json() == broker_student
    else:
        assert response.status_code == 200, response.__dict__
        assert response.content == b""


@pytest.mark.parametrize("method,status_code", [("POST", 201), ("PUT", 200)])
def test_put_post_user(func_client, method, status_code, broker_student, monkeypatch, async_iterable):
    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.get_user_by_id: kelvin_student,
            kelvin_session: get_preconfigured_session_mock(
                {
                    "post.return_value": kelvin_student(),
                }
            ),
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    monkeypatch.setattr(
        users,
        "_check_user_dependencies",
        AsyncMock(),
    )
    monkeypatch.setattr(
        users,
        "get_pseudonyms_for_providers",
        async_iterable,
    )
    url = f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/users"
    query = None
    if method == "POST":
        query = _client.post
    if method == "PUT":
        query = _client.put
        url += f"/{broker_student['id']}"
    response = query(url, json=broker_student)
    assert response.status_code == status_code
    assert response.json() == broker_student


def test_post_conflict_kelvin_request(func_client, monkeypatch, broker_student, async_iterable):
    async def post_mock(*args, **kwargs):
        raise InvalidRequest(msg="test", status=409, reason="test-reason")

    async def reload_mock(*args, **kwargs):
        return

    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.get_user_by_id: kelvin_student,
            kelvin_session: get_preconfigured_session_mock({"post": post_mock}),
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    monkeypatch.setattr(
        users.KelvinUser,
        "reload",
        reload_mock,
    )
    monkeypatch.setattr(
        users,
        "_check_user_dependencies",
        AsyncMock(),
    )
    monkeypatch.setattr(
        users,
        "get_pseudonyms_for_providers",
        async_iterable,
    )
    url = f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/users"
    response = _client.post(url, json=broker_student)
    assert response.status_code == 409
    assert response.json() == {"detail": {"conflict_id": broker_student["id"], "msg": "test-reason"}}


def test_delete_user(func_client, broker_student):
    _client = func_client(
        custom_overrides={
            provisioning_plugin.dependencies.get_user_by_id: kelvin_student,
            provisioning_plugin.dependencies.check_for_authority_admin: lambda: True,
        }
    )
    response = _client.delete(f"/v1/{DEFAULT_SCHOOL_AUTHORITY}/users/{broker_student['id']}")
    assert response.status_code == 204
