import uuid
import zlib

import pytest
import requests

from id_broker_common.utils import ldap_settings
from provisioning_plugin.models import School
from ucsschool.kelvin.client import SchoolResource

from ..utils import (
    DEFAULT_SCHOOL_AUTHORITY,
    DEFAULT_SCHOOL_NAME,
    EXPECTED_HEADER_NAME,
    must_run_in_docker,
)

pytestmark = must_run_in_docker


@pytest.mark.asyncio
async def test_schools_get(create_school, url_fragment, auth_headers):
    school = await create_school()
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.get(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/"
        f"schools/{school.udm_properties.get('ucsschoolRecordUID')}",
        headers=headers,
    )
    assert response.status_code == 200, response.__dict__
    assert response.json() == {
        "name": DEFAULT_SCHOOL_NAME,
        "display_name": school.display_name,
        "id": school.udm_properties.get("ucsschoolRecordUID"),
    }, response.__dict__
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_schools_get_wrong_auth(create_school, url_fragment, auth_headers):
    await create_school(DEFAULT_SCHOOL_AUTHORITY)
    await create_school("TEST2")
    headers = await auth_headers("TEST2")
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.get(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/schools/{DEFAULT_SCHOOL_NAME}",
        headers=headers,
    )
    assert response.status_code == 403, response.__dict__
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_schools_post(url_fragment, auth_headers, kelvin_session_obj, faker):
    school_name = faker.pystr(1, 7)
    school_id = faker.pystr(1, 7)
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.post(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/schools",
        data=School(name=school_name, display_name="TEST DISPLAY NAME", id=school_id).json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 201, response.__dict__
    school = await SchoolResource(session=kelvin_session_obj).get(
        name=f"{DEFAULT_SCHOOL_AUTHORITY}-{school_name}"
    )
    assert school.name == f"{DEFAULT_SCHOOL_AUTHORITY}-{school_name}"
    assert school.display_name == "TEST DISPLAY NAME"
    dc_name = f"dc-{zlib.crc32(DEFAULT_SCHOOL_AUTHORITY.encode('UTF-8'))}"
    assert school.home_share_file_server == dc_name
    assert school.class_share_file_server == dc_name
    assert school.educational_servers == [dc_name]
    assert school.udm_properties.get("ucsschoolRecordUID") == school_id
    assert school.udm_properties.get("ucsschoolSourceUID") == DEFAULT_SCHOOL_AUTHORITY
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_schools_delete(
    create_school,
    url_fragment,
    auth_headers,
    create_school_class,
    create_workgroup,
    create_user,
    wait_for_dn_exists,
):
    school = await create_school()
    dns_in_school = []
    for _ in range(5):
        school_class = await create_school_class()
        workgroup = await create_workgroup()
        user = await create_user(
            school_classes={DEFAULT_SCHOOL_NAME: [school_class.name]},
            workgroups={DEFAULT_SCHOOL_NAME: [workgroup.name]},
        )
        dns_in_school.extend([user.dn, school_class.dn, workgroup.dn])

    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.delete(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/"
        f"schools/{school.udm_properties.get('ucsschoolRecordUID')}",
        headers=headers,
    )
    assert response.status_code == 204, response.__dict__
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id
    await wait_for_dn_exists(dn=school.dn, exists=False)
    ldap_base = ldap_settings().ldap_base
    dns_in_school.extend(
        [
            f"cn=OU{school.name}-Member-Verwaltungsnetz,cn=ucsschool,cn=groups,{ldap_base}",
            f"cn=OU{school.name}-Member-Edukativnetz,cn=ucsschool,cn=groups,{ldap_base}",
            f"cn=OU{school.name}-Klassenarbeit,cn=ucsschool,cn=groups,{ldap_base}",
            f"cn=OU{school.name}-DC-Verwaltungsnetz,cn=ucsschool,cn=groups,{ldap_base}",
            f"cn=OU{school.name}-DC-Edukativnetz,cn=ucsschool,cn=groups,{ldap_base}",
            f"cn=admins-{school.name},cn=ouadmins,cn=groups,{ldap_base}",
        ]
    )
    # Instead of waiting a long time we should expect that everything
    # is removed after the ou is not existing anymore.
    for dn in dns_in_school:
        await wait_for_dn_exists(dn=dn, exists=False, timeout=1)
    response = requests.get(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/"
        f"schools/{school.udm_properties.get('ucsschoolRecordUID')}",
        headers=headers,
    )
    assert response.status_code == 404
