import uuid

import pytest
import requests

from provisioning_plugin.models import SchoolClass
from ucsschool.kelvin.client import NoObject, SchoolClassResource

from ..utils import (
    DEFAULT_SCHOOL_AUTHORITY,
    DEFAULT_SCHOOL_NAME,
    EXPECTED_HEADER_NAME,
    must_run_in_docker,
)

pytestmark = must_run_in_docker


@pytest.mark.asyncio
async def test_school_classes_get(
    create_school, create_school_class, create_user, url_fragment, auth_headers
):
    await create_school()
    user = await create_user()
    school_class = await create_school_class(descr="DESCR", members=[user.name])
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.get(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/"
        f"classes/{school_class.udm_properties.get('ucsschoolRecordUID')}",
        headers=headers,
    )
    assert response.status_code == 200, response.__dict__
    assert response.json() == {
        "name": school_class.name,
        "school": DEFAULT_SCHOOL_NAME,
        "members": [user.record_uid],
        "description": "DESCR",
        "id": school_class.udm_properties.get("ucsschoolRecordUID"),
    }, response.__dict__
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_school_classes_head(
    create_school, create_school_class, create_user, url_fragment, auth_headers
):
    await create_school()
    user = await create_user()
    school_class = await create_school_class(descr="DESCR", members=[user.name])
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.head(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/"
        f"classes/{school_class.udm_properties.get('ucsschoolRecordUID')}",
        headers=headers,
    )
    assert response.status_code == 200, response.__dict__
    assert response.content == b""
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_school_classes_get_wrong_auth(
    create_school, create_school_class, url_fragment, auth_headers
):
    await create_school(DEFAULT_SCHOOL_AUTHORITY)
    await create_school("TEST2")
    headers = await auth_headers("TEST2")
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    school_class = await create_school_class(descr="DESCR", members=[])
    response = requests.get(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/"
        f"classes/{school_class.udm_properties.get('ucsschoolRecordUID')}",
        headers=headers,
    )
    assert response.status_code == 403, response.__dict__
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_school_classes_post(
    create_school, create_user, url_fragment, auth_headers, kelvin_session_obj, faker, wait_for_dn_exists
):
    await create_school()
    u1 = await create_user()
    u2 = await create_user(roles=["student"])
    school_class_name = faker.pystr(1, 7)
    school_class_id = faker.pystr(1, 7)
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.post(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/classes/",
        data=SchoolClass(
            id=school_class_id,
            name=school_class_name,
            description="DESCR",
            school=DEFAULT_SCHOOL_NAME,
            members={u1.record_uid, u2.record_uid},
        ).json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 201, response.__dict__
    school_class = await SchoolClassResource(session=kelvin_session_obj).get(
        name=school_class_name, school=f"{DEFAULT_SCHOOL_AUTHORITY}-{DEFAULT_SCHOOL_NAME}"
    )
    assert school_class.name == school_class_name
    assert school_class.description == "DESCR"
    assert school_class.school == f"{DEFAULT_SCHOOL_AUTHORITY}-{DEFAULT_SCHOOL_NAME}"
    assert set(school_class.users) == {u1.name, u2.name}
    assert school_class.udm_properties.get("ucsschoolRecordUID") == school_class_id
    assert school_class.udm_properties.get("ucsschoolSourceUID") == DEFAULT_SCHOOL_AUTHORITY
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id
    await wait_for_dn_exists(school_class.dn)
    # The kelvin client does not return the expected value.
    # assert school_class.create_share is False
    dn = school_class.dn.replace("cn=klassen", "cn=klassen,cn=shares")
    await wait_for_dn_exists(dn, timeout=20, exists=False)


@pytest.mark.asyncio
async def test_school_classes_put(
    create_school, create_user, create_school_class, url_fragment, auth_headers, kelvin_session_obj
):
    await create_school()
    u1 = await create_user()
    u2 = await create_user(roles=["student"])
    school_class = await create_school_class(descr="DESCR")
    new_school_class = SchoolClass.without_prefix(
        await SchoolClass.from_kelvin_school_class(school_class.as_dict(), session=kelvin_session_obj),
        DEFAULT_SCHOOL_AUTHORITY,
    )
    new_school_class.members = {u1.record_uid, u2.record_uid}
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.put(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/classes/"
        f"{school_class.udm_properties.get('ucsschoolRecordUID')}",
        data=new_school_class.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 200, response.__dict__
    kelvin_school_class = SchoolClass.from_kelvin_school_class(
        (
            await SchoolClassResource(session=kelvin_session_obj).get(
                name=school_class.name, school=school_class.school
            )
        ).as_dict(),
        session=kelvin_session_obj,
    )
    assert await kelvin_school_class == SchoolClass.with_prefix(
        new_school_class, DEFAULT_SCHOOL_AUTHORITY
    )
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_school_classes_change_school_fails_put(
    create_school, create_user, create_school_class, url_fragment, auth_headers, kelvin_session_obj
):
    await create_school()
    await create_school(name="otherschool")
    u1 = await create_user()
    u2 = await create_user(roles=["student"])
    school_class = await create_school_class(descr="DESCR")
    new_school_class = SchoolClass.without_prefix(
        await SchoolClass.from_kelvin_school_class(school_class.as_dict(), session=kelvin_session_obj),
        DEFAULT_SCHOOL_AUTHORITY,
    )
    new_school_class.school = "otherschool"
    new_school_class.members = {u1.record_uid, u2.record_uid}
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.put(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/classes/"
        f"{school_class.udm_properties.get('ucsschoolRecordUID')}",
        data=new_school_class.json(),
        headers={"Content-Type": "application/json", **headers},
    )
    assert response.status_code == 422, response.__dict__
    assert response.json() == {"detail": "The school must not be changed after creation."}
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id


@pytest.mark.asyncio
async def test_school_classes_delete(
    create_school, create_school_class, url_fragment, auth_headers, kelvin_session_obj
):
    await create_school()
    school_class = await create_school_class(descr="DESCR")
    headers = await auth_headers(DEFAULT_SCHOOL_AUTHORITY)
    request_id = uuid.uuid4().hex
    headers[EXPECTED_HEADER_NAME] = request_id
    response = requests.delete(
        f"{url_fragment}/provisioning/v1/{DEFAULT_SCHOOL_AUTHORITY}/"
        f"classes/{school_class.udm_properties.get('ucsschoolRecordUID')}",
        headers=headers,
    )
    assert response.status_code == 204, response.__dict__
    with pytest.raises(NoObject):
        await SchoolClassResource(session=kelvin_session_obj).get(
            name=school_class.name, school=f"{DEFAULT_SCHOOL_AUTHORITY}-{DEFAULT_SCHOOL_NAME}"
        )
    assert EXPECTED_HEADER_NAME in response.headers
    assert response.headers[EXPECTED_HEADER_NAME] == request_id
