#!/usr/share/ucs-test/runner pytest-3 -s -l -v
## -*- coding: utf-8 -*-
## desc: Check id broker settings and cmdline tool
## tags: [ucsschool-id-broker]
## packages: [id-broker-common]
## exposure: dangerous

import json
import random

import pytest
from utils import _env_is_not_empty, add_test_service, remove_test_service

import univention.testing.strings as uts
import univention.testing.ucsschool.ucs_test_school as utu
from ucsschool.lib.models.utils import exec_cmd
from univention.id_broker.pseudonyms import generate_pseudonym
from univention.id_broker.pseudonyms_udm_host import get_settings_data_content

pytestmark = pytest.mark.skipif(
    _env_is_not_empty(),
    reason="This test assumes and environment without provisioned users, school_classes and ous.",
)


@pytest.mark.parametrize(
    "setting,settings_dn",
    [
        ("secrets", "secrets_settings_data_dn"),
        ("mappings", "mappings_settings_data_dn"),
    ],
)
def test_add_remove_setting(udm, setting, settings_dn, preserve_settings_data, request):
    settings_dn = request.getfixturevalue(settings_dn)
    preserve_settings_data(settings_dn)

    key = uts.random_string()
    i = random.randint(1, 30)
    value = f"idBrokerPseudonym{str(i).zfill(4)}"
    expected_data = {key: value}
    exec_cmd(
        [
            "manage-service-providers",
            "addsetting",
            setting,
            "--key",
            key,
            "--value",
            value,
        ]
    )
    rv, stdout, stderr = exec_cmd(["manage-service-providers", "showsetting", setting])
    res = json.loads(stdout)
    assert expected_data == res

    ldap_data = udm.obj_by_dn(settings_dn)
    ldap_data = json.loads(ldap_data.props.data.raw)

    assert expected_data == ldap_data
    assert ldap_data == res

    expected_data = {}
    exec_cmd(
        [
            "manage-service-providers",
            "removesetting",
            setting,
            "--key",
            key,
        ]
    )
    rv, stdout, stderr = exec_cmd(["manage-service-providers", "showsetting", setting])
    res = json.loads(stdout)
    assert expected_data == res

    ldap_data = udm.obj_by_dn(settings_dn)
    ldap_data = json.loads(ldap_data.props.data.raw)

    assert expected_data == ldap_data
    assert ldap_data == res


def test_add_remove_service_provider(
    udm, mappings_settings_data_dn, secrets_settings_data_dn, preserve_settings_data
):
    preserve_settings_data(mappings_settings_data_dn)
    preserve_settings_data(secrets_settings_data_dn)
    school_auth = uts.random_string()
    with utu.UCSTestSchool() as schoolenv:
        ou_name, ou_dn = schoolenv.create_ou(name_edudc=schoolenv.ucr["hostname"])
        student_name, student_dn = schoolenv.create_user(ou_name)
        workgroup_name, workgroup_dn = schoolenv.create_workgroup(ou_name)
        schoolclass_name, schoolclass_dn = schoolenv.create_school_class(ou_name)

        dn_list = [ou_dn, student_dn, workgroup_dn, schoolclass_dn]
        for obj_dn in dn_list:
            if obj_dn != workgroup_dn:
                obj = udm.obj_by_dn(obj_dn)
                obj.props.ucsschoolSourceUID = school_auth
                obj.props.ucsschoolRecordUID = uts.random_string()
                obj.save()
            assert obj.props.idBrokerPseudonym0001 is None

        mappings = get_settings_data_content(mappings_settings_data_dn)
        assert mappings == {}
        secrets = get_settings_data_content(secrets_settings_data_dn)
        assert secrets == {}

        service_provider = uts.random_string()
        secret = uts.random_string()
        add_test_service(service_provider, secret)

        for obj_dn in dn_list:
            provisioned_obj = udm.obj_by_dn(obj_dn)
            # not for workgroups
            if obj_dn != workgroup_dn:
                obj_record_uid = provisioned_obj.props.ucsschoolRecordUID
                obj_authority = provisioned_obj.props.ucsschoolSourceUID
                pseudonym = generate_pseudonym(secret, obj_record_uid, obj_authority)
                assert provisioned_obj.props.idBrokerPseudonym0001 == pseudonym
            else:
                assert provisioned_obj.props.idBrokerPseudonym0001 is None

        mappings = get_settings_data_content(mappings_settings_data_dn)
        assert mappings == {service_provider: "idBrokerPseudonym0001"}
        secrets = get_settings_data_content(secrets_settings_data_dn)
        assert secrets == {service_provider: secret}
        remove_test_service(service_provider)

        for obj_dn in dn_list:
            provisioned_obj = udm.obj_by_dn(obj_dn)
            assert provisioned_obj.props.idBrokerPseudonym0001 is None

        mappings = get_settings_data_content(mappings_settings_data_dn)
        assert mappings == {}
        secrets = get_settings_data_content(secrets_settings_data_dn)
        assert secrets == {}


def test_add_same_service_provider_abort(
    udm, mappings_settings_data_dn, secrets_settings_data_dn, preserve_settings_data
):
    preserve_settings_data(mappings_settings_data_dn)
    preserve_settings_data(secrets_settings_data_dn)
    service_provider = uts.random_string()
    secret = uts.random_string()
    assert add_test_service(service_provider, secret)[0] == 0
    assert add_test_service(service_provider, secret)[0] == 1


def test_add_same_service_provider_overwrite(
    udm, mappings_settings_data_dn, secrets_settings_data_dn, preserve_settings_data
):
    preserve_settings_data(mappings_settings_data_dn)
    preserve_settings_data(secrets_settings_data_dn)
    service_provider = uts.random_string()
    secret = uts.random_string()
    secret2 = uts.random_string()
    assert add_test_service(service_provider, secret)[0] == 0
    mappings = get_settings_data_content(mappings_settings_data_dn)
    assert mappings == {service_provider: "idBrokerPseudonym0001"}
    secrets = get_settings_data_content(secrets_settings_data_dn)
    assert secrets == {service_provider: secret}
    assert add_test_service(service_provider, secret2, overwrite=True)[0] == 0
    mappings = get_settings_data_content(mappings_settings_data_dn)
    assert mappings == {service_provider: "idBrokerPseudonym0001"}
    secrets = get_settings_data_content(secrets_settings_data_dn)
    assert secrets == {service_provider: secret2}
