import logging
import random
from typing import Dict, List, Tuple

from faker import Faker
from locust import events, task

from performance_tests.common import (
    ProvisioningClient,
    ResponseError,
    TestData,
    UsedSchools,
    get_classes,
    get_provisioning_api_admins,
)

school_authorities: List[str] = []  # ["Traeger1", "Traeger"]
user_credentials: Dict[str, Tuple[str, str]] = {}  # [(username, password), ..]
id_broker_school_classes: Dict[str, Dict[str, Dict[str, str]]] = {}  # {sch_auth: {ou: {cl: uuid, ...}}}
fake = Faker()
NUM_SCHOOLS_PER_SCHOOL_AUTHORITY = 5  # loading all OUs from test DB takes to long
used_schools = UsedSchools()


@events.init.add_listener
def on_init(environment, **kwargs):
    data = TestData()
    school_authorities.extend(data.school_authorities)
    user_credentials.update(get_provisioning_api_admins(data))
    id_broker_school_classes.update(get_classes(data, NUM_SCHOOLS_PER_SCHOOL_AUTHORITY))


class ProvisioningAPIClient(ProvisioningClient):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.school_authority = random.choice(school_authorities)
        logging.info("Testing with school authority: %r", self.school_authority)
        self.auth_username, self.auth_password = user_credentials[self.school_authority]
        self.schools = list(id_broker_school_classes[self.school_authority].keys())
        school_classes = sum(len(v) for v in id_broker_school_classes[self.school_authority].values())
        logging.info("Testing with schools: %r incl. %d classes.", sorted(self.schools), school_classes)

    @task
    def school_class_update(self) -> None:
        school = random.choice(self.schools)
        used_schools.add(self.school_authority, school)
        classes = id_broker_school_classes[self.school_authority][school]
        class_name, entry_uuid = random.choice(list(classes.items()))
        try:
            super().school_class_update(
                school_authority=self.school_authority,
                obj_id=entry_uuid,
                name=class_name,
                school=school,
                description=f"Performance Test temp class modified {fake.pyint(1000, 9999)}",
                members=[],
            )
        except ResponseError as exc:
            logging.error(
                "%s: school_authority=%r name=%r entry_uuid=%r -> %r",
                self.greenlet.name,
                self.school_authority,
                class_name,
                entry_uuid,
                exc,
            )


@events.test_stop.add_listener
def _(environment, **_kwargs):
    logging.info("Used schools: %r", sorted(f"{s_a}-{ou}" for s_a, ou in used_schools.schools))
    used_schools.persist()
