import logging
import random
from typing import Dict, List, Tuple

from locust import events, task

from performance_tests.common import (
    ProvisioningClient,
    ResponseError,
    TestData,
    get_classes,
    get_provisioning_api_admins,
)

school_authorities: List[str] = []  # ["Traeger1", "Traeger"]
user_credentials: Dict[str, Tuple[str, str]] = {}  # [(username, password), ..]
id_broker_school_classes: Dict[str, Dict[str, Dict[str, str]]] = {}  # {sch_auth: {ou: {cl: uuid, ...}}}
NUM_SCHOOLS_PER_SCHOOL_AUTHORITY = 5  # loading all OUs from test DB takes to long


@events.init.add_listener
def on_init(environment, **kwargs):
    data = TestData()
    school_authorities.extend(data.school_authorities)
    user_credentials.update(get_provisioning_api_admins(data))
    id_broker_school_classes.update(get_classes(data, NUM_SCHOOLS_PER_SCHOOL_AUTHORITY))


class ProvisioningAPIClient(ProvisioningClient):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.school_authority = random.choice(school_authorities)
        logging.info("Testing with school authority: %r", self.school_authority)
        self.auth_username, self.auth_password = user_credentials[self.school_authority]
        self.schools = list(id_broker_school_classes[self.school_authority].keys())
        school_classes = sum(len(v) for v in id_broker_school_classes[self.school_authority].values())
        logging.info("Testing with schools: %r incl. %d classes.", sorted(self.schools), school_classes)

    @task
    def school_class_head(self) -> None:
        school = random.choice(self.schools)
        classes = id_broker_school_classes[self.school_authority][school]
        name, entry_uuid = random.choice(list(classes.items()))
        try:
            super().school_class_exists(school_authority=self.school_authority, obj_id=entry_uuid)
        except ResponseError as exc:
            logging.error(
                "%s: school_authority=%r name=%r entry_uuid=%r -> %r",
                self.greenlet.name,
                self.school_authority,
                name,
                entry_uuid,
                exc,
            )
