# -*- coding: utf-8 -*-

# Copyright 2022 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

import logging
import os
from typing import Any, Dict, Iterable, List, Union
from urllib.parse import urljoin

import aiohttp
from ldap3.utils.conv import escape_filter_chars
from starlette import status
from ucsschool_id_connector_defaults.group_handler_base import (
    GroupDispatcherPluginBase,
    GroupNotFoundError,
    PerSchoolAuthorityGroupDispatcherBase,
)
from ucsschool_id_connector_defaults.user_handler_base import (
    PerSchoolAuthorityUserDispatcherBase,
    UserDispatcherPluginBase,
    UserNotFoundError,
)

from idbroker.id_broker_client import (
    IDBrokerError,
    IDBrokerNotFoundError,
    IDBrokerSchool,
    IDBrokerSchoolClass,
    IDBrokerUser,
    IDBrokerWorkGroup,
    School,
    SchoolClass,
    User,
    WorkGroup,
)
from idbroker.utils import (
    get_configured_schools,
    get_role_info,
    group_dn_has_configured_school,
    role_has_configured_school,
    school_in_configured_schools,
    user_has_left_configured_schools,
    user_has_unconfigured_schools,
)
from ucsschool_id_connector.ldap_access import LDAPAccess
from ucsschool_id_connector.models import (
    ListenerGroupAddModifyObject,
    ListenerGroupRemoveObject,
    ListenerObject,
    ListenerUserAddModifyObject,
    ListenerUserRemoveObject,
    SchoolAuthorityConfiguration,
)
from ucsschool_id_connector.plugins import hook_impl, plugin_manager
from ucsschool_id_connector.utils import school_class_dn_regex, workgroup_dn_regex

ContextDictType = Dict[str, Dict[str, Iterable[str]]]

SchoolGroupType = Union[SchoolClass, WorkGroup]


async def ping_id_broker(school_authority: SchoolAuthorityConfiguration) -> bool:
    """
    To ping the id-broker side, we try to login and get a token.
    """
    url = urljoin(school_authority.url, "ucsschool/apis/auth/token")
    verify_ssl = "UNSAFE_SSL" not in os.environ
    id_broker_conf = school_authority.plugin_configs["id_broker"]
    payload = {
        "username": id_broker_conf["username"],
        "password": id_broker_conf["password"].get_secret_value(),
    }
    async with aiohttp.ClientSession(trust_env=True) as session:
        try:
            async with session.post(url, ssl=verify_ssl, data=payload) as resp:
                if resp.status != 200:
                    return False
        except aiohttp.client_exceptions.ClientConnectorError:
            return False
    return True


async def create_school_if_missing(
    ou: str, ldap_access: LDAPAccess, id_broker_school: IDBrokerSchool, logger: logging.Logger
) -> None:
    entries = await ldap_access.search(
        filter_s=f"(&(objectClass=ucsschoolOrganizationalUnit)(ou={escape_filter_chars(ou)}))",
        attributes=["displayName", "entryUUID"],
    )
    if len(entries) == 1:
        entry_uuid = entries[0].entryUUID.value
        display_name = entries[0].displayName.value
        if not await id_broker_school.exists(entry_uuid):
            logger.info("Creating school %r...", ou)
            await id_broker_school.create(School(id=entry_uuid, name=ou, display_name=display_name))
    else:
        raise IDBrokerNotFoundError(404, f"School {ou!r} does not exist on sender system.")


async def create_class_if_missing(
    school_class: str,
    ou: str,
    ldap_access: LDAPAccess,
    id_broker_school_class: IDBrokerSchoolClass,
    logger: logging.Logger,
) -> None:
    entries = await ldap_access.search(
        filter_s=f"(&(objectClass=univentionGroup)(cn={escape_filter_chars(f'{ou}-{school_class}')}))",
        attributes=["description", "entryUUID"],
    )
    if len(entries) == 1:
        entry_uuid = entries[0].entryUUID.value
        description = entries[0].description.value
        if not await id_broker_school_class.exists(entry_uuid):
            logger.info("Creating school class %r in school %r...", school_class, ou)
            await id_broker_school_class.create(
                SchoolClass(
                    id=entry_uuid, name=school_class, description=description, school=ou, members=[]
                )
            )
    else:
        raise IDBrokerNotFoundError(
            404, f"School class {school_class!r} in school {ou!r} does not exist on sender system."
        )


async def create_work_group_if_missing(
    work_group: str,
    ou: str,
    ldap_access: LDAPAccess,
    id_broker_work_group: IDBrokerWorkGroup,
    logger: logging.Logger,
) -> None:
    entries = await ldap_access.search(
        filter_s=f"(&(objectClass=univentionGroup)(cn={escape_filter_chars(f'{ou}-{work_group}')}))",
        attributes=["description", "entryUUID"],
    )
    if len(entries) == 1:
        entry_uuid = entries[0].entryUUID.value
        description = entries[0].description.value
        if not await id_broker_work_group.exists(entry_uuid):
            logger.info("Creating school class %r in school %r...", work_group, ou)
            await id_broker_work_group.create(
                WorkGroup(id=entry_uuid, name=work_group, description=description, school=ou, members=[])
            )
    else:
        raise IDBrokerNotFoundError(
            404, f"Workgroup {work_group!r} in school {ou!r} does not exist on sender system."
        )


def initial_mode_is_activated(school_authority: SchoolAuthorityConfiguration) -> bool:
    return school_authority.plugin_configs["id_broker"].get("initial_import_mode", False) is True


class UcsschoolRoleStringError(Exception):
    pass


class UnknownRole(UcsschoolRoleStringError):
    pass


class UnknownContextType(UcsschoolRoleStringError):
    pass


class InvalidUcsschoolRoleString(UcsschoolRoleStringError):
    pass


role_workgroup: str = "workgroup"


#
# Users
#


class IDBrokerPerSAUserDispatcher(PerSchoolAuthorityUserDispatcherBase):
    _required_search_params = ("id", "school_authority")

    def __init__(self, school_authority: SchoolAuthorityConfiguration, plugin_name: str):
        super(IDBrokerPerSAUserDispatcher, self).__init__(school_authority, plugin_name)
        self.class_dn_regex = school_class_dn_regex()
        self.workgroup_dn_regex = workgroup_dn_regex()
        self.attribute_mapping = {
            "id": "id",
            "username": "user_name",
            "firstname": "first_name",
            "lastname": "last_name",
            "context": "context",
        }
        self.id_broker_school_class = IDBrokerSchoolClass(self.school_authority, "id_broker")
        self.id_broker_work_group = IDBrokerWorkGroup(self.school_authority, "id_broker")
        self.id_broker_school = IDBrokerSchool(self.school_authority, "id_broker")
        self.id_broker_user = IDBrokerUser(self.school_authority, "id_broker")
        self.ldap_access = LDAPAccess()

    @property
    def initial_import_mode(self):
        return initial_mode_is_activated(self.school_authority)

    async def create_or_update_preconditions_met(self, obj: ListenerUserAddModifyObject) -> bool:
        return True

    async def print_ids(self, obj: ListenerUserAddModifyObject) -> None:
        self.logger.info(f"Object that is being created or updated: {obj}")

    async def _handle_attr_context(self, obj: ListenerUserAddModifyObject) -> Dict[str, Any]:
        # Add primary school 1st to context dict (which is always ordered in cpython3). It will then be
        # the 1st in the requests json (at least with the OpenAPI client we're using) and will then
        # hopefully become the primary school at the ID Broker. That is not strictly necessary, as the
        # Self-Disclosure API does not expose that, but it'll be less confusing, when comparing users.
        schools = sorted(obj.schools)
        schools.remove(obj.school)
        schools.insert(0, obj.school)
        context = {
            school_name: {"roles": set(obj.school_user_roles), "classes": set(), "workgroups": set()}
            for school_name in schools
        }
        for group_dn in obj.object.get("groups", []):
            m = self.class_dn_regex.match(group_dn)
            if m:
                name = m.groupdict()["name"]
                school = m.groupdict()["ou"]
                context[school]["classes"].add(name)
                continue
            m = self.workgroup_dn_regex.match(group_dn)
            if m:
                name = m.groupdict()["name"]
                school = m.groupdict()["ou"]
                context[school]["workgroups"].add(name)
        return context

    async def _handle_attr_id(self, obj: ListenerUserAddModifyObject) -> str:
        return obj.id

    async def search_params(
        self, obj: Union[ListenerUserAddModifyObject, ListenerUserRemoveObject]
    ) -> Dict[str, Any]:
        return {"school_authority": self.school_authority.name, "id": obj.id}

    async def fetch_obj(self, search_params: Dict[str, Any]) -> User:
        """
        Retrieve a user from ID Broker API.

        If it does not exist on the id broker, we need to raise an UserNotFoundError, so it will be
        created.
        """
        self.logger.debug("Retrieving user with search parameters: %r", search_params)
        try:
            return await self.id_broker_user.get(obj_id=search_params["id"])
        except IDBrokerNotFoundError:
            raise UserNotFoundError(f"No user found with search params: {search_params!r}.")

    async def _create_schools_and_school_groups_if_missing(self, context: ContextDictType) -> None:
        for school in sorted(context):
            await create_school_if_missing(school, self.ldap_access, self.id_broker_school, self.logger)
            for school_class in context[school]["classes"]:
                await create_class_if_missing(
                    school_class, school, self.ldap_access, self.id_broker_school_class, self.logger
                )
            for work_group in context[school]["workgroups"]:
                await create_work_group_if_missing(
                    work_group, school, self.ldap_access, self.id_broker_work_group, self.logger
                )

    def _empty_groups(self, context: ContextDictType) -> ContextDictType:
        self.logger.debug("Empty classes of user for faster initial import mode.")
        for school in context:
            context[school]["classes"] = []
            context[school]["workgroups"] = []
        return context

    async def do_create(self, request_body: Dict[str, Any]) -> None:
        """Create a user object at the target."""
        self.logger.info(
            "Going to create user %r (%r)...", request_body["user_name"], request_body["id"]
        )
        self.logger.debug("request_body=%r", request_body)
        if self.initial_import_mode:
            request_body["context"] = self._empty_groups(request_body["context"])
        await self._create_schools_and_school_groups_if_missing(request_body["context"])
        user: User = await self.id_broker_user.create(User(**request_body))
        self.logger.info("User created: %r.", user)

    async def do_modify(self, request_body: Dict[str, Any], api_user_data: User) -> None:
        """Modify a user object at the target."""
        self.logger.info("Going to modify user %r (%r)...", api_user_data.user_name, api_user_data.id)
        self.logger.debug("request_body=%r", request_body)
        if self.initial_import_mode:
            request_body["context"] = self._empty_groups(request_body["context"])
        await self._create_schools_and_school_groups_if_missing(request_body["context"])
        user: User = await self.id_broker_user.update(User(**request_body))
        self.logger.info("User modified: %r.", user)

    async def do_remove(self, obj: ListenerGroupRemoveObject, api_user_data: User) -> None:
        """Delete a user object at the target."""
        self.logger.info("Going to delete user: %r...", obj)
        await self.id_broker_user.delete(api_user_data.id)
        self.logger.info("User deleted: %r.", api_user_data)


class IDBrokerUserDispatcher(UserDispatcherPluginBase):
    plugin_name = "id_broker-users"
    per_s_a_handler_class = IDBrokerPerSAUserDispatcher

    @hook_impl
    async def school_authority_ping(self, school_authority: SchoolAuthorityConfiguration) -> bool:
        """impl for ucsschool_id_connector.plugins.Postprocessing.school_authority_ping"""
        if not await ping_id_broker(school_authority):
            self.logger.error(
                "Failed to call ucsschool-api for school authority API (%s)",
                school_authority.name,
            )
            return False
        self.logger.info(
            "Successfully called ucsschool-api for school authority API (%s)",
            school_authority.name,
        )
        return True

    @hook_impl
    async def handle_listener_object(
        self, school_authority: SchoolAuthorityConfiguration, obj: ListenerObject
    ) -> bool:
        if isinstance(obj, ListenerUserAddModifyObject):
            configured_schools = get_configured_schools(school_authority)
            if user_has_left_configured_schools(obj, configured_schools):
                # User was not deleted, but has no configured schools left
                # -> delete user instead of modifying it.
                self.logger.debug(f"{obj} got removed from all configured schools")
                await self.handler(school_authority, self.plugin_name).handle_remove(obj)
                return True
            if user_has_unconfigured_schools(obj, configured_schools):
                # Remove unconfigured schools from user
                self.logger.debug(f"{obj} has schools unconfigured schools")
                obj.object["school"] = [
                    school
                    for school in obj.object["school"]
                    if school_in_configured_schools(school.lower(), configured_schools)
                ]
                obj.object["ucsschoolRole"] = [
                    role_string
                    for role_string in obj.object["ucsschoolRole"]
                    if role_has_configured_school(role_string, configured_schools)
                ]
                obj.object["groups"] = [
                    group_dn
                    for group_dn in obj.object["groups"]
                    if group_dn_has_configured_school(group_dn, configured_schools)
                ]
        return await super().handle_listener_object(school_authority, obj)


#
# Groups
#


def has_workgroup_role(roles: List[str], school: str) -> bool:
    for r in roles:
        role, context_type, context = get_role_info(r)
        if role == role_workgroup and context_type == "school" and context == school:
            return True
    return False


class IDBrokerPerSAGroupDispatcher(PerSchoolAuthorityGroupDispatcherBase):
    _required_search_params = ("id", "school_authority")

    def __init__(self, school_authority: SchoolAuthorityConfiguration, plugin_name: str):
        super(IDBrokerPerSAGroupDispatcher, self).__init__(school_authority, plugin_name)
        self.attribute_mapping = {
            "id": "id",
            "name": "name",
            "description": "description",
            "school": "school",
            "users": "members",
            "ucsschoolRole": "ucsschoolRole",
        }
        self.id_broker_work_group = IDBrokerWorkGroup(self.school_authority, "id_broker")
        self.id_broker_school = IDBrokerSchool(self.school_authority, "id_broker")
        self.id_broker_school_class = IDBrokerSchoolClass(self.school_authority, "id_broker")
        self.id_broker_user = IDBrokerUser(self.school_authority, "id_broker")
        self.class_dn_regex = school_class_dn_regex()
        self.work_group_dn_regex = workgroup_dn_regex()
        self.ldap_access = LDAPAccess()

    @property
    def initial_import_mode(self):
        return initial_mode_is_activated(self.school_authority)

    async def create_or_update_preconditions_met(self, obj: ListenerGroupAddModifyObject) -> bool:
        """Verify preconditions for creating or modifying object on target."""
        return (
            await self.is_schools_class(obj) or await self.is_work_group(obj)
        ) and not self.initial_import_mode

    async def remove_preconditions_met(self, obj: ListenerGroupRemoveObject) -> bool:
        """
        Verify preconditions for removing object on target.
        """
        return (
            await self.is_schools_class(obj) or await self.is_work_group(obj)
        ) and not self.initial_import_mode

    async def is_schools_class(
        self, obj: Union[ListenerGroupAddModifyObject, ListenerGroupRemoveObject]
    ) -> bool:
        """Check if group is a school class."""
        return bool(self.class_dn_regex.match(obj.dn))

    async def is_work_group(
        self, obj: Union[ListenerGroupAddModifyObject, ListenerGroupRemoveObject]
    ) -> bool:
        """Check if group is a work group."""
        return bool(self.work_group_dn_regex.match(obj.dn))

    async def search_params(
        self, obj: Union[ListenerGroupAddModifyObject, ListenerGroupRemoveObject]
    ) -> Dict[str, Any]:
        return {"school_authority": self.school_authority.name, "id": obj.id, "dn": obj.dn}

    async def fetch_obj(self, search_params: Dict[str, Any]) -> SchoolGroupType:
        """
        Retrieve a school group (class or work group) from ID Broker API.

        :return SchoolGroupType: school group object on the ID Broker
        :raises GroupNotFoundError: if it does not exist on the ID Broker
        """
        self.logger.debug("Retrieving school group with search parameters: %r", search_params)
        m = self.class_dn_regex.match(search_params["dn"])
        if m:
            try:
                return await self.id_broker_school_class.get(obj_id=search_params["id"])
            except (IDBrokerNotFoundError, IDBrokerError):
                raise GroupNotFoundError(f"No school class found with search params: {search_params!r}.")
        else:
            try:
                return await self.id_broker_work_group.get(obj_id=search_params["id"])
            except (IDBrokerNotFoundError, IDBrokerError):
                raise GroupNotFoundError(f"No workgroup found with search params: {search_params!r}.")

    async def _handle_attr_id(self, obj: ListenerUserAddModifyObject) -> str:
        return obj.id

    async def _handle_attr_school(self, obj: ListenerGroupAddModifyObject) -> str:
        """Name of school for this school group on the target."""
        m = self.class_dn_regex.match(obj.dn)
        if not m:
            m = self.work_group_dn_regex.match(obj.dn)
        return m.groupdict()["ou"]

    async def _handle_attr_users(self, obj: ListenerGroupAddModifyObject) -> List[str]:
        """
        User dns of a school group have to be entryUUID of the users.
        Hint: In the user plugin the record_uid of the user is set to entryUUID on the id-broker side.
        Here we want to do this again. The record_uid of the user is a different value.
        """
        record_uids = []
        for dn in obj.users:
            user_entries = await self.ldap_access.search(
                base=dn, filter_s="(objectClass=*)", attributes=["entryUUID"]
            )
            if len(user_entries) > 1:
                self.logger.warning(f"Member {dn!r} of group {obj.name!r} was found multiple times.")
            elif len(user_entries) < 1:
                self.logger.warning(f"Member {dn!r} of group {obj.name!r} was not found.")
            else:
                record_uids.append(str(user_entries[0].entryUUID))
        return sorted(record_uids)

    async def _handle_attr_name(self, obj: ListenerGroupAddModifyObject) -> str:
        """The name should not include the school prefix. It is prepended on the id-broker side."""
        sc_m = self.class_dn_regex.match(obj.dn)
        wg_m = self.work_group_dn_regex.match(obj.dn)
        if sc_m or wg_m:
            m = sc_m or wg_m
            school = m.groupdict()["ou"]
            prefix = f"{school}-"
            if obj.name.startswith(prefix):
                return obj.name[len(prefix) :]
        return obj.name

    async def do_create(self, request_body: Dict[str, Any]) -> None:
        """Create a school group object at the target."""
        name, school = request_body["name"], request_body["school"]
        if self.initial_import_mode:
            self.logger.debug(
                "Skipping creation of school group %r in  %r in initial import mode.", name, school
            )
            return
        self.logger.info(
            "Going to create school group %r in school %r: %r...",
            name,
            school,
            request_body,
        )
        await create_school_if_missing(school, self.ldap_access, self.id_broker_school, self.logger)
        try:
            if has_workgroup_role(request_body["ucsschoolRole"], request_body["school"]):
                g = WorkGroup(**request_body)
                self.logger.info("Creating work group %s", g)
                await self.id_broker_work_group.create(g)
            else:
                g = SchoolClass(**request_body)
                self.logger.info("Creating school class %s", g)
                await self.id_broker_school_class.create(g)
        except IDBrokerError as exc:
            if exc.status == status.HTTP_422_UNPROCESSABLE_ENTITY and "member" in str(exc).lower():
                self.logger.error("Unknown member(s) when creating school group %r: %s", name, exc)
                await self._fix_school_group_members(g)
            else:
                raise
        self.logger.info("School group created: %r.", g)

    async def do_modify(
        self, request_body: Dict[str, Any], api_school_group_data: SchoolGroupType
    ) -> None:
        """Modify a school group object at the target."""
        name, school = request_body["name"], request_body["school"]
        if self.initial_import_mode:
            self.logger.debug(
                "Skipping modification of school group %r in %r in initial import mode.", name, school
            )
            return
        self.logger.info(
            "Going to modify school group %r in %r: %r...",
            name,
            school,
            request_body,
        )

        try:
            if has_workgroup_role(request_body["ucsschoolRole"], request_body["school"]):
                g = WorkGroup(**request_body)
                self.logger.info("Updating work group %s", g)
                await self.id_broker_work_group.update(g)
            else:
                g = SchoolClass(**request_body)
                self.logger.info("Updating school class %s", g)
                await self.id_broker_school_class.update(g)
        except IDBrokerError as exc:
            if exc.status == status.HTTP_422_UNPROCESSABLE_ENTITY and "member" in str(exc).lower():
                self.logger.error(
                    "Unknown member(s) when updating school group %r: %s",
                    api_school_group_data.name,
                    exc,
                )
                await self._fix_school_group_members(g)
            else:
                raise
        self.logger.info(
            "School group modified: %r  in school %r: %r...",
            name,
            school,
            request_body,
        )

    async def do_remove(
        self, obj: ListenerGroupRemoveObject, api_school_group_data: SchoolGroupType
    ) -> None:
        """Delete a school group object at the target."""
        self.logger.info("Going to delete school group: %r...", obj)
        m = self.class_dn_regex.match(obj.dn)
        try:
            if m:
                await self.id_broker_school_class.delete(api_school_group_data.id)
            else:
                await self.id_broker_work_group.delete(api_school_group_data.id)
        except (IDBrokerNotFoundError, IDBrokerError):
            self.logger.warning(f"No school group found with id: {api_school_group_data.id!r}.")
        self.logger.info("Group deleted: %r.", api_school_group_data)

    async def _fix_school_group_members(self, school_group: SchoolGroupType):
        self.logger.info("Trying to fix members of %r...", school_group)
        # 1st check if school group exists
        try:
            if isinstance(school_group, SchoolClass):
                await self.id_broker_school_class.get(obj_id=school_group.id)
            else:
                await self.id_broker_work_group.get(obj_id=school_group.id)
        except (IDBrokerNotFoundError, IDBrokerError):
            self.logger.info("School group does %r not exists, creating...", school_group)
            members = school_group.members.copy()
            school_group.members.clear()
            if isinstance(school_group, SchoolClass):
                school_group = await self.id_broker_school_class.create(school_group)
            else:
                school_group = await self.id_broker_work_group.create(school_group)
            school_group.members.extend(members)
        # now let's add as many members as possible
        self.logger.debug("Verifying existence of members of %r...", school_group)
        new_members = []
        for user_id in school_group.members:
            try:
                user: User = await self.id_broker_user.get(obj_id=user_id)
                self.logger.debug("Found member of %r at ID Broker server: %r.", school_group, user)
                new_members.append(user_id)
            except IDBrokerNotFoundError:
                self.logger.error("Missing member of %r at ID Broker server: %r.", school_group, user_id)
                filter_s = f"(&(objectClass=posixAccount)(entryUUID={user_id}))"
                results = await self.ldap_access.search(filter_s=filter_s)
                if len(results) == 1:
                    self.logger.error("Missing member DN in local LDAP is: %r", results[0].entry_dn)
                else:
                    self.logger.error(
                        "Could not find user in local LDAP when searching with %r.", filter_s
                    )
        if new_members:
            self.logger.info(
                "Setting the following members for %r at ID Broker: %r. Omitted members: %r",
                school_group,
                new_members,
                sorted(set(school_group.members) - set(new_members)),
            )
            school_group.members.clear()
            school_group.members.extend(new_members)
            if isinstance(school_group, SchoolClass):
                await self.id_broker_school_class.update(school_group)
            else:
                await self.id_broker_work_group.update(school_group)


class IDBrokerGroupDispatcher(GroupDispatcherPluginBase):
    plugin_name = "id_broker-groups"
    per_s_a_handler_class = IDBrokerPerSAGroupDispatcher

    @hook_impl
    async def school_authority_ping(self, school_authority: SchoolAuthorityConfiguration) -> bool:
        """impl for ucsschool_id_connector.plugins.Postprocessing.school_authority_ping"""
        if not await ping_id_broker(school_authority):
            self.logger.error(
                "Failed to call ucsschool-api for school authority API (%s)",
                school_authority.name,
            )
            return False
        self.logger.info(
            "Successfully called ucsschool-api for school authority API (%s)",
            school_authority.name,
        )
        return True


plugin_manager.register(IDBrokerUserDispatcher(), IDBrokerUserDispatcher.plugin_name)
plugin_manager.register(IDBrokerGroupDispatcher(), IDBrokerGroupDispatcher.plugin_name)
