#!/usr/bin/env python3
#
# Copyright 2022-2023 Univention GmbH
#
# http://www.univention.de/
#
# All rights reserved.
#
# The source code of this program is made available
# under the terms of the GNU Affero General Public License version 3
# (GNU AGPL V3) as published by the Free Software Foundation.
#
# Binary versions of this program provided by Univention to you as
# well as other copyrighted, protected or trademarked materials like
# Logos, graphics, fonts, specific documentations and configurations,
# cryptographic keys etc. are subject to a license agreement between
# you and Univention and not subject to the GNU AGPL V3.
#
# In the case you use this program under the terms of the GNU AGPL V3,
# the program is provided in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License with the Debian GNU/Linux or Univention distribution in file
# /usr/share/common-licenses/AGPL-3; if not, see
# <http://www.gnu.org/licenses/>.

import argparse
import sys
from urllib.parse import urljoin

import requests
from keycloak import KeycloakAdmin
from keycloak.exceptions import KeycloakGetError

from univention.config_registry import ucr


def get_parser():
    parser = argparse.ArgumentParser(description="Add ID Broker client in Keycloak")
    subparsers = parser.add_subparsers(dest="subcommand", help="Choose a client type")
    subparsers.required = True
    parser_oidc = subparsers.add_parser(
        "oidc",
        help="oidc type client (prefered type if your Keycloak instance is publicly available)",
    )
    parser_oidc.add_argument(
        "--secret",
        help="client secret (will be provided to you)",
        type=str,
        required=True,
    )
    parser_oidc.set_defaults(func=oidc_keycloak)
    parser_saml = subparsers.add_parser(
        "saml", help="saml type client (if your Keycloak instance is not publicly available)"
    )
    parser_saml.add_argument(
        "--get-idp-metadata",
        help="IDP metadata from your Keycloak instance",
        action="store_true",
    )
    parser_saml.set_defaults(func=saml_keycloak)
    parser.add_argument(
        "--keycloak-fqdn",
        help="fqdn to your keycloak instance",
        default=ucr.get("keycloak/server/sso/fqdn", f"ucs-sso-ng.{ucr.get('domainname')}"),
        type=str,
    )
    parser.add_argument(
        "--keycloak-path",
        help="path to your keycloak instance",
        default=ucr.get("keycloak/server/sso/path"),
        type=str,
    )
    parser.add_argument(
        "--id-broker-fqdn",
        help="fqdn to the ID Broker",
        default="sso-broker.production.univention-id-broker.com",
        type=str,
    )
    parser.add_argument(
        "--authority",
        help="Your school authority name on the ID Broker",
        type=str,
        required=True,
    )
    return parser


def oidc_keycloak(args):
    kca = get_keycloak_client(args)
    create_oidc_client(kca, args.id_broker_fqdn, args.authority, args.secret)


def saml_keycloak(args):
    kca = get_keycloak_client(args)
    if args.get_idp_metadata:
        saml_metadata_url = urljoin(kca.server_url, "realms/ucs/protocol/saml/descriptor")
        resp = requests.get(saml_metadata_url)
        print(resp.text)
    else:
        create_saml_client(kca, args.id_broker_fqdn, args.authority)


def create_oidc_client(kca, id_broker_fqdn, authority, secret):
    client_config = {
        "clientId": "id-broker",
        "secret": secret,
        "name": "ID Broker",
        "description": "ID Broker",
        "enabled": True,
        "clientAuthenticatorType": "client-secret",
        "redirectUris": [f"https://{id_broker_fqdn}/auth/realms/ID-Broker/broker/{authority}/endpoint"],
        "webOrigins": ["+"],
        "consentRequired": False,
        "implicitFlowEnabled": False,
        "directAccessGrantsEnabled": False,
        "serviceAccountsEnabled": False,
        "publicClient": False,
        "protocol": "openid-connect",
        "attributes": {
            "post.logout.redirect.uris": "+",
            "backchannel.logout.url": f"https://{id_broker_fqdn}/auth/realms/ID-Broker/"
            f"broker/{authority}/endpoint",
            "frontchannel.logout.url": f"https://{id_broker_fqdn}/auth/realms/ID-Broker/"
            f"broker/{authority}/endpoint",
            "use.jwks.url": "true",
            "jwks.url": f"https://{id_broker_fqdn}/auth/realms/ID-Broker/protocol/openid-connect/certs",
            "pkceEnabled": True,
            "pkceMethod": "S256",
            "validateSignature": True,
        },
    }
    entryUUID_mapper = {
        "config": {
            "access.token.claim": True,
            "claim.name": "sub",
            "id.token.claim": "true",
            "introspection.token.claim": True,
            "jsonType.label": "String",
            "user.attribute": "entryUUID",
            "userinfo.token.claim": True,
        },
        "consentRequired": False,
        "name": "entryUUID2sub",
        "protocol": "openid-connect",
        "protocolMapper": "oidc-usermodel-attribute-mapper",
    }
    try:
        kca.create_client(client_config)
    except KeycloakGetError as exc:
        print(exc, file=sys.stderr)
        sys.exit(1)
    kc_client_id = kca.get_client_id(client_config["clientId"])
    scopes = kca.get_client_scopes()
    for scope in scopes:
        if scope["name"] in ["profile", "email"]:
            kca.raw_delete(
                f"/admin/realms/ucs/clients/{kc_client_id}/default-client-scopes/{scope['id']}"
            )
            break
    kca.add_mapper_to_client(kc_client_id, entryUUID_mapper)
    print("OIDC client has been created", file=sys.stderr)


def create_saml_client(kca, id_broker_fqdn, authority):
    resp = requests.get(
        f"https://{id_broker_fqdn}/auth/realms/ID-Broker/broker/{authority}/endpoint/descriptor"
    )
    try:
        resp.raise_for_status()
    except requests.exceptions.HTTPError as exc:
        print(exc, file=sys.stderr)
        sys.exit(1)
    id_broker_metadata = resp.text
    entryUUID_mapper = {
        "protocol": "saml",
        "protocolMapper": "saml-user-attribute-mapper",
        "name": "entryUUID",
        "config": {
            "user.attribute": "entryUUID",
            "friendly.name": "entryUUID",
            "attribute.name": "entryUUID",
            "attribute.nameformat": "Basic",
        },
    }
    client_config = kca.raw_post(
        "/admin/realms/ucs/client-description-converter", id_broker_metadata
    ).json()
    client_config["name"] = "ID Broker"
    try:
        kca.create_client(client_config)
    except KeycloakGetError as exc:
        print(exc, file=sys.stderr)
        sys.exit(1)
    kc_client_id = kca.get_client_id(client_config["clientId"])
    kca.add_mapper_to_client(kc_client_id, entryUUID_mapper)
    print("SAML client has been created", file=sys.stderr)


def get_keycloak_client(args):
    kc_secret = open("/etc/keycloak.secret", "r").read().strip()
    realm = "ucs"
    keycloak_url = urljoin(f"https://{args.keycloak_fqdn}", args.keycloak_path)
    kca = KeycloakAdmin(
        server_url=keycloak_url,
        username="admin",
        password=kc_secret,
        realm_name=realm,
        user_realm_name="master",
    )
    return kca


def main():
    args = get_parser().parse_args()
    args.func(args)


if __name__ == "__main__":
    main()
