import asyncio
import time
from functools import lru_cache
from pathlib import Path
from typing import Any, Dict, Iterator, List, Optional, TypeVar

from async_property import async_property
from fastapi import HTTPException, status

from id_broker_common.utils import load_settings
from ucsschool.apis.utils import get_correlation_id, get_logger
from ucsschool.kelvin.client import (
    KelvinClientError,
    KelvinObject,
    KelvinResource,
    NoObject,
    Session,
    User,
    UserResource,
)
from ucsschool.kelvin.client.session import Token

kelvin_session_kwargs: Dict[str, Any] = {}


PARALLELISM_DEFAULT = 4
KELVIN_SESSION_DEFAULT_TIMEOUT = 180
T = TypeVar("T")


class IDBrokerKelvinSession(Session):
    """
    We need to override these methods to be able to set the
    request id for each request while keeping the same session.
    """

    @async_property
    async def json_headers(self) -> Dict[str, str]:
        return {
            "accept": "application/json",
            "Authorization": f"Bearer {await self.token}",
            "Content-Type": "application/json",
            "Access-Control-Expose-Headers": "X-Request-ID",
            "X-Request-ID": get_correlation_id(),
        }

    @async_property
    async def token(self) -> str:
        if not self._token or not self._token.is_valid():
            resp_json = await self.post(
                self.urls["token"],
                headers={
                    "Content-Type": "application/x-www-form-urlencoded",
                    "Access-Control-Expose-Headers": "X-Request-ID",
                    "X-Request-ID": get_correlation_id(),
                },
                data={"username": self.username, "password": self.password},
            )
            self._token = Token.from_str(resp_json["access_token"])
        return self._token.value


_kelvin_session: Optional[IDBrokerKelvinSession] = None


def load_kelvin_session_kwargs(path: Path) -> Dict[str, str]:
    json_settings = load_settings(path)
    try:
        # only return required settings
        return {
            "username": json_settings["username"],
            "password": json_settings["password"],
            "host": json_settings["host"],
            "verify": json_settings["verify_ssl"],
        }
    except KeyError as exc:
        raise ValueError(f"Incomplete or malformed settings file '{path!s}': {exc!s}") from exc


@lru_cache(maxsize=1)
def kelvin_on_primary_session_kwargs(path: Path) -> Dict[str, str]:
    json_settings = load_settings(path)
    try:
        # only return required settings
        return {
            "username": json_settings["username"],
            "password": json_settings["password"],
            "host": json_settings["host_primary"],
            "verify": json_settings["verify_ssl"],
        }
    except KeyError as exc:
        raise ValueError(f"Incomplete or malformed settings file '{path!s}': {exc!s}") from exc


async def kelvin_session() -> Session:
    # this function must be async, or Session.__init__() will fail because there is not yet an event loop
    global _kelvin_session
    if not _kelvin_session:
        _kelvin_session = IDBrokerKelvinSession(
            **kelvin_session_kwargs, timeout=KELVIN_SESSION_DEFAULT_TIMEOUT
        )
        _kelvin_session.open()
    return _kelvin_session


async def get_kelvin_obj(resource: KelvinResource, name: str, school: str = None) -> KelvinObject:
    """
    Retrieve an object via the Kelvin API.
    This is much faster than searching, if `school` and `name` are known (for `SchoolResource` and
    `UserResource` the `school` argument can be omitted).

    :param resource: The KelvinResource to use for retrieving the object
    :param str school: the OU of the object
    :param str name: the name of the object
    :return: The found KelvinObject
    :raises HTTPException: If the request failed or no object was found
    """
    logger = get_logger()
    get_kwargs = {"name": name}
    if school is not None:
        get_kwargs["school"] = school
    try:
        return await resource.get(**get_kwargs)
    except NoObject as exc:
        logger.debug("%s not found with %r.", resource.Meta.kelvin_object.__name__, get_kwargs)
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"{resource.Meta.kelvin_object.__name__} not found.",
        ) from exc
    except KelvinClientError as exc:
        logger.error(
            "Received non-404 KelvinClientError when retrieving %s with %r: %s",
            resource.Meta.kelvin_object.__name__,
            get_kwargs,
            exc,
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"Internal error retrieving {resource.Meta.kelvin_object.__name__}.",
        ) from exc


async def search_kelvin_obj(resource: KelvinResource, **search_args) -> KelvinObject:
    """
    Searches for an object via the Kelvin API and returns it.

    :param resource: The KelvinResource to use for searching for the object
    :param search_args: search arguments passed to the resources search method
    :return: The found KelvinObject
    :raises HTTPException: If the request failed, if no object or if more than one object was found
    """
    logger = get_logger()
    try:
        kelvin_objects = [kelvin_object async for kelvin_object in resource.search(**search_args)]
    except KelvinClientError as exc:
        raise HTTPException(status_code=exc.status, detail=exc.reason) from exc
    if not kelvin_objects:
        logger.debug(
            "%s not found with search params: %r.", resource.Meta.kelvin_object.__name__, search_args
        )
        raise HTTPException(
            status_code=status.HTTP_404_NOT_FOUND,
            detail=f"{resource.Meta.kelvin_object.__name__} not found.",
        )
    elif len(kelvin_objects) > 1:
        logger.error(
            "Multiple %s objects found with search params: %r",
            resource.Meta.kelvin_object.__name__,
            search_args,
        )
        raise HTTPException(
            status_code=status.HTTP_500_INTERNAL_SERVER_ERROR,
            detail=f"More than one {resource.Meta.kelvin_object.__name__} was found.",
        )
    return kelvin_objects[0]


@lru_cache(maxsize=2)
def kelvin_access_parallelism_setting(path: Path) -> int:
    json_settings = load_settings(path)
    try:
        return json_settings["kelvin_access_parallelism"]
    except KeyError:
        return PARALLELISM_DEFAULT


def split_list(li: List[T], chunk_size: int) -> Iterator[List[T]]:
    """Split a list into chunks of equal size (except last chunk, which is <= `chunk_size`)."""
    len_li = len(li)
    for i in range(0, len_li, chunk_size):
        yield li[i : i + chunk_size]


async def retrieve_users_in_parallel(
    user_names: List[str], *, parallelism: int, session: Session
) -> List[User]:
    """Retrieve multiple Kelvin user objects in parallel."""
    logger = get_logger()
    users: List[User] = []
    t0 = time.time()
    for user_name_list in split_list(user_names, parallelism):
        users.extend(
            await asyncio.gather(
                *[
                    get_kelvin_obj(UserResource(session=session), name=user_name)
                    for user_name in user_name_list
                ]
            )
        )
    t1 = time.time()
    logger.debug(
        "Retrieved %d users (parallelism: %d) from Kelvin in %.3f seconds.",
        len(user_names),
        parallelism,
        t1 - t0,
    )
    return users
